#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq
from pyramid.httpexceptions import HTTPFound, HTTPInternalServerError, HTTPNotFound
from pyramid.response import Response
from pyramid.view import render_view_to_response, view_config
from zope.interface import Interface
from zope.traversing.interfaces import ITraversable

from onf_website.reference.forest.interfaces import IForestTable
from onf_website.reference.forest.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.forest.model.foret import Foret, InformationForet, ProprietaireForet
from onf_website.reference.hunting.model import GestionDate
from onf_website.shared.hunting import FakeHuntingCalendar, HUNTING_CONTENT_TYPE, IHuntingManager, \
    MissingHuntingCalendar
from pyams_alchemy.engine import get_user_session
from pyams_catalog.query import CatalogResultSet
from pyams_skin.interfaces import ISkinnable
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.skin import apply_skin
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces.url import DISPLAY_CONTEXT
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url, canonical_url
from pyams_workflow.interfaces import IWorkflow, IWorkflowVersions

__docformat__ = 'restructuredtext'


@adapter_config(name='cc',
                context=(Interface, IPyAMSUserLayer),
                provides=ITraversable)
class HuntingCalendarTraverser(ContextRequestAdapter):
    """Hunting calendar namespace traverser"""

    def traverse(self, name, furtherpath=None):
        manager = query_utility(IHuntingManager)
        if manager is None:
            raise HTTPNotFound()
        if not name:
            raise HTTPNotFound()
        forest_id = name.upper()
        if '::' in forest_id:
            forest_id, ignored = forest_id.split('::', 1)
            if not forest_id:
                raise HTTPNotFound()
        request = self.request
        # look for matching forest
        session = get_user_session(RDF_SESSION)
        ref_foret, prop, info = session.query(Foret,
                                              ProprietaireForet,
                                              InformationForet) \
            .join(ProprietaireForet, Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
            .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .filter(Foret.id_nat_frt == forest_id) \
            .order_by(ProprietaireForet.pourcentage_part.desc(), ProprietaireForet.categorie) \
            .first() or (None, None, None)
        if not ref_foret:
            raise HTTPNotFound()
        if info and not info.visible:
            raise HTTPNotFound()
        table = query_utility(IForestTable)
        if (table is None) or (not table.hunting_displays.get(prop.categorie)):
            raise HTTPNotFound()
        # look for Artemis calendar data
        has_dates = GestionDate.get_dates(forest_id, valid_only=True).count()
        if has_dates:
            return FakeHuntingCalendar(forest_id)
        # look for shared hunting calendar
        workflow = IWorkflow(manager)
        catalog = get_utility(ICatalog)
        params = Eq(catalog['content_type'], HUNTING_CONTENT_TYPE) & \
            Any(catalog['workflow_state'], workflow.visible_states) & \
            Eq(catalog['forests'], forest_id)
        has_results, results = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
        if has_results:
            calendar = next(results)
            if calendar is not None:
                location = canonical_url(calendar, request, query=request.params)
                if location == request.url:
                    request.annotations[DISPLAY_CONTEXT] = request.context
                    request.context = calendar
                    response = render_view_to_response(calendar, request, '')
                    return response
                raise HTTPFound(location=location)
        # try to fallback with missing hunting calendar content
        return MissingHuntingCalendar(forest_id)


@view_config(route_name='hunting_access')
def get_hunting_access(request):
    """Direct access to given hunting calendar"""
    manager = query_utility(IHuntingManager)
    if manager is None:
        raise HTTPNotFound()
    forest_id = request.matchdict.get('id_nat_frt')
    if not forest_id:
        raise HTTPNotFound()
    if '::' in forest_id:
        forest_id, ignored = forest_id.split('::', 1)
        if not forest_id:
            raise HTTPNotFound()
    forest_id = forest_id.upper()
    # look for content matching given forest ID
    view_name = ''.join(request.matchdict.get('view'))
    session = get_user_session(RDF_SESSION)
    ref_foret, prop, info = session.query(Foret,
                                          ProprietaireForet,
                                          InformationForet) \
        .join(ProprietaireForet, Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
        .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
        .filter(Foret.id_nat_frt == forest_id) \
        .order_by(ProprietaireForet.pourcentage_part.desc(), ProprietaireForet.categorie) \
        .first() or (None, None, None)
    if not ref_foret:
        raise HTTPNotFound()
    if info and not info.visible:
        raise HTTPNotFound()
    table = query_utility(IForestTable)
    if (table is None) or (not table.hunting_displays.get(prop.categorie)):
        raise HTTPNotFound()
    # look for Artemis calendar data
    has_dates = GestionDate.get_dates(forest_id).count()
    if has_dates:
        fake_calendar = FakeHuntingCalendar(forest_id)
        fake_calendar.forest_ids = [forest_id]
        target = fake_calendar.__parent__
        site = get_parent(target, ISkinnable)
        apply_skin(request, ISkinnable(site).get_skin(request))
        request.annotations[DISPLAY_CONTEXT] = target
        request.context = fake_calendar
        return render_view_to_response(fake_calendar, request, view_name)
    # look for shared hunting calendar
    workflow = IWorkflow(manager, None)
    catalog = get_utility(ICatalog)
    params = Eq(catalog['content_type'], HUNTING_CONTENT_TYPE) & \
        Any(catalog['workflow_state'], workflow.visible_states) & \
        Eq(catalog['forests'], forest_id)
    has_results, results = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
    if has_results:
        calendar = next(results)
        if calendar is not None:
            if view_name:
                location = absolute_url(calendar, request, view_name)
            else:
                location = canonical_url(calendar, request, query=request.params)
            if location == request.url:
                request.annotations[DISPLAY_CONTEXT] = request.context
                request.context = calendar
                response = render_view_to_response(calendar, request, view_name)
            else:
                response = Response()
                response.status_code = HTTPFound.code
                response.location = location
            return response
    # missing calendar fallback
    fake_calendar = MissingHuntingCalendar(forest_id)
    fake_calendar.forest_ids = [forest_id]
    target = fake_calendar.__parent__
    site = get_parent(target, ISkinnable)
    apply_skin(request, ISkinnable(site).get_skin(request))
    request.annotations[DISPLAY_CONTEXT] = target
    request.context = fake_calendar
    return render_view_to_response(fake_calendar, request, view_name)
