# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyams_content.component.map.interfaces import have_gis
if have_gis:

    import json
    
    from hypatia.catalog import CatalogQuery
    from hypatia.interfaces import ICatalog
    from hypatia.query import And, Eq, Any
    from pyramid.decorator import reify
    from zope.interface import Interface
    from zope.schema.fieldproperty import FieldProperty
    
    from onf_website.component.map.portlet import IMapPortletSettings
    from onf_website.component.map.portlet.skin import MapPortletDefaultRenderer, MapPortletDefaultRendererSettings
    from onf_website.reference.hunting.model import GestionDate, SaisonChasse
    from onf_website.shared.hunting import HUNTING_CONTENT_TYPE, IHuntingManager, IWfHuntingCalendar
    from onf_website.shared.hunting.skin.interfaces import IHuntingCalendarLocationMapPortletRendererSettings
    from pyams_catalog.query import CatalogResultSet
    from pyams_portal.interfaces import IPortalContext, IPortletRenderer
    from pyams_skin.layer import IPyAMSLayer
    from pyams_template.template import template_config
    from pyams_utils.adapter import adapter_config
    from pyams_utils.factory import factory_config
    from pyams_utils.registry import get_utility
    from pyams_workflow.interfaces import IWorkflow
    
    from onf_website import _
    
    
    @factory_config(IHuntingCalendarLocationMapPortletRendererSettings)
    class HuntingCalendarLocationMapPortletRendererSettings(MapPortletDefaultRendererSettings):
        """Hunting calendar location map portlet renderer settings"""
        
        display_content_link = FieldProperty(
                IHuntingCalendarLocationMapPortletRendererSettings['display_content_link'])
        content_link_label = FieldProperty(
                IHuntingCalendarLocationMapPortletRendererSettings['content_link_label'])

        
    @adapter_config(name='hunting-calendar',
                    required=(IPortalContext, IPyAMSLayer, Interface, IMapPortletSettings),
                    provides=IPortletRenderer)
    @template_config(template='templates/map-calendar.pt', layer=IPyAMSLayer)
    class MapPortletHuntingCalendarRenderer(MapPortletDefaultRenderer):
        """Map portlet hunting calendar renderer"""
        
        label = _("ONF: Hunting calendar location map")
        weight = 20
        
        settings_interface = IHuntingCalendarLocationMapPortletRendererSettings

        def render(self, template_name=''):
            hunting_calendar = IWfHuntingCalendar(self.context, None)
            if hunting_calendar is None:
                return ''
            return super().render(template_name)

        @reify
        def hunting_manager(self):
            return get_utility(IHuntingManager)
        
        @reify
        def current_saison(self):
            return SaisonChasse.get_current_or_next()
        
        def get_forest_ids(self):
            return json.dumps(self.context.forest_ids)
        
        def get_calendar_dates(self):
            result = []
            for date in GestionDate.get_dates(self.context.forest_ids, valid_only=True):
                result.append({
                    'date': date.date_chasse.date().isoformat(),
                    'id_contrat': date.id_nat_contrat,
                    'id_lots': [
                        attribution.id_nat_lot
                        for attribution in date.contrat.attributions
                    ],
                    'has_perimetres': all((
                        len(attribution.lot.perimetres)
                        for attribution in date.contrat.attributions
                    ))
                })
            return json.dumps(result)

        def get_last_date(self):
            saison = self.current_saison
            if saison is not None:
                return saison.date_fin_saison.isoformat()

        def has_unvalidated_dates(self):
            return GestionDate.get_unvalidated_dates(self.context.forest_ids).count() > 0

        def get_link_targets(self):
            catalog = get_utility(ICatalog)
            manager = self.hunting_manager
            workflow = get_utility(IWorkflow, name=manager.shared_content_workflow)
            params = And(Eq(catalog['content_type'], HUNTING_CONTENT_TYPE),
                         Any(catalog['forests'], self.context.forest_ids),
                         Any(catalog['workflow_state'], workflow.visible_states))
            return list(CatalogResultSet(CatalogQuery(catalog).query(params)))
            