#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing.interfaces import IHearingManagerTarget
from onf_website.component.location import ILocationManagerTarget
from onf_website.shared.common.security import SharedToolManagerRestrictionInfo
from onf_website.shared.hunting import HuntingCalendar
from onf_website.shared.hunting.interfaces import HUNTING_CONTENT_TYPE, IHuntingCalendarInfo, \
    IHuntingManager, IHuntingManagerFactory
from pyams_content.component.paragraph import IParagraphFactorySettings
from pyams_content.component.theme import ITagsTarget, IThemesManagerTarget
from pyams_content.reference.pictograms.interfaces import IPictogramManagerTarget
from pyams_content.shared.common import ISharedSite
from pyams_content.shared.common.interfaces import IManagerRestrictionsFactory, \
    ISharedContentFactory
from pyams_content.shared.common.manager import SharedTool
from pyams_sequence.reference import get_reference_target
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import query_utility, utility_config


__docformat__ = 'restructuredtext'

from pyams_utils.traversing import get_parent


@implementer(IHuntingManager, IParagraphFactorySettings, ILocationManagerTarget,
             IHearingManagerTarget, IPictogramManagerTarget, IThemesManagerTarget,
             ITagsTarget)
class HuntingManager(SharedTool):
    """Hunting manager persistent class"""

    shared_content_type = HUNTING_CONTENT_TYPE
    shared_content_types_fields = IHuntingCalendarInfo

    allowed_paragraphs = FieldProperty(IParagraphFactorySettings['allowed_paragraphs'])
    auto_created_paragraphs = FieldProperty(IParagraphFactorySettings['auto_created_paragraphs'])
    
    notepad = FieldProperty(IHuntingManager['notepad'])
    title_prefix = FieldProperty(IHuntingManager['title_prefix'])
    season_prefix = FieldProperty(IHuntingManager['season_prefix'])
    default_header = FieldProperty(IHuntingManager['default_header'])
    default_hearing_source = FieldProperty(IHuntingManager['default_hearing_source'])
    default_source_folder = FieldProperty(IHuntingManager['default_source_folder'])

    default_template = FieldProperty(IHuntingManager['default_template'])
    missing_info_template = FieldProperty(IHuntingManager['missing_info_template'])
    
    msg_generic_info = FieldProperty(IHuntingManager['msg_generic_info'])
    msg_incomplete_info = FieldProperty(IHuntingManager['msg_incomplete_info'])
    msg_missing_info = FieldProperty(IHuntingManager['msg_missing_info'])
    msg_no_hunting_info = FieldProperty(IHuntingManager['msg_no_hunting_info'])

    def get_default_source(self):
        """Get default hearing source"""
        target = get_reference_target(self.default_source_folder)
        if target is None:
            target = query_utility(ISharedSite, name=self.default_hearing_source)
        return target


@utility_config(provides=IHuntingManagerFactory)
class HuntingManagerFactory:
    """Hunting manager factory"""

    def __new__(cls):
        return HuntingManager


@adapter_config(context=IHuntingManager, provides=IManagerRestrictionsFactory)
def hunting_manager_restrictions_factory(context):
    """Hunting manager custom restrictions factory"""
    return SharedToolManagerRestrictionInfo


@adapter_config(context=IHuntingManager, provides=ISharedContentFactory)
def hunting_manager_content_factory(context):
    """Hunting manager content factory"""
    return HuntingCalendar


@subscriber(IObjectAddedEvent, context_selector=IHuntingManager)
def handle_added_hunting_manager(event):
    """Register hunting manager when added"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IHuntingManager)
