#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import And, Any, Eq
from sqlalchemy.sql import and_, func, literal_column, or_
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from onf_website.reference.forest import IForestTable
from onf_website.reference.forest.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.forest.model.foret import Foret, ForetGeom, ForetSurCommune, \
    InformationForet, ProprietaireForet, REG_ID_NAT_FRT
from onf_website.reference.insee.model import CommuneGeom, PARENT_SESSION as INSEE_SESSION, \
    REG_COMMUNE_CODE
from onf_website.shared.forest import FOREST_CONTENT_TYPE, FakeForest
from onf_website.shared.hub.interfaces import IForestHubSearchFolder, IForestHubSearchFolderQuery
from pyams_alchemy.engine import get_user_session
from pyams_catalog.query import CatalogResultSet
from pyams_content.component.illustration import IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.search import ISearchFolderRoles
from pyams_content.shared.view import WfView
from pyams_content.workflow import VISIBLE_STATES
from pyams_gis.interfaces import WGS84, WGS84WM
from pyams_gis.transform import transform
from pyams_portal.interfaces import IPortalContext
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request


__docformat__ = 'restructuredtext'

from onf_website import _


@implementer(IForestHubSearchFolder, ISearchFolderRoles, IIllustrationTarget,
             ILinkIllustrationTarget, IPortalContext, IPreviewTarget)
class ForestHubSearchFolder(WfView):
    """Forest hub search folder"""

    content_name = _("Forests search folder")

    handle_short_name = True
    handle_header = True
    handle_description = True

    sequence_name = ''
    sequence_prefix = ''

    forests_map_template = FieldProperty(IForestHubSearchFolder['forests_map_template'])
    forests_list_template = FieldProperty(IForestHubSearchFolder['forests_list_template'])
    activities_map_template = FieldProperty(IForestHubSearchFolder['activities_map_template'])
    activities_list_template = FieldProperty(IForestHubSearchFolder['activities_list_template'])

    @staticmethod
    def is_deletable():
        return True

    def get_results(self, context, sort_index=None, reverse=None, limit=None,
                    start=0, length=None, ignore_cache=False, get_count=False, request=None,
                    aggregates=None, settings=None, **kwargs):
        if request is None:
            request = check_request()
        content_type = request.params.get('content_type', FOREST_CONTENT_TYPE)
        search_adapter = request.registry.queryMultiAdapter((context, request),
                                                            IForestHubSearchFolderQuery,
                                                            name=content_type)
        if search_adapter is not None:
            return search_adapter.get_results(context, sort_index, reverse, limit,
                                              start, length, ignore_cache,
                                              get_count, request, aggregates, settings, **kwargs)
        return super().get_results(context, sort_index, reverse, limit, start, length,
                                   ignore_cache, get_count, request, aggregates, settings, **kwargs)


@adapter_config(name=FOREST_CONTENT_TYPE,
                context=(IForestHubSearchFolder, IPyAMSLayer),
                provides=IForestHubSearchFolderQuery)
class ForestHubSearchFolderForestSearchAdapter(ContextRequestAdapter):
    """Forest hub search folder forests search adapter"""

    def get_results(self, context, sort_index=None, reverse=None, limit=None,
                    start=0, length=None, ignore_cache=False, get_count=False, request=None,
                    aggregates=None, settings=None, **kwargs):
        results, count, aggregations = [], 0, {}
        if request is None:
            request = check_request()
        user_search = request.params.get('user_search', '').strip()
        map_center = request.params.get('map_center')
        if not (user_search or map_center):
            return (results, count, aggregations) if get_count else results
        # get visible forests categories
        table = query_utility(IForestTable)
        params = [
            ProprietaireForet.categorie.in_(table.visible_owners_ids),
            or_(InformationForet.visible == None,
                InformationForet.visible == True)
        ]
        # get query map center
        session = get_user_session(RDF_SESSION)
        geo_marker = None
        if map_center:
            target = transform(map_center.split(), WGS84, WGS84WM)
            marker = 'SRID=3857;POINT({} {})'.format(target['point']['longitude'],
                                                     target['point']['latitude'])
            geo_marker = session.scalar(func.ST_GeomFromText(marker))
        else:
            # Forest or commune search
            if REG_ID_NAT_FRT.match(user_search.upper()):
                session = get_user_session(RDF_SESSION)
                geo_marker = session.query(
                        func.ST_AsEWKT(
                            func.ST_Transform(
                                func.ST_PointOnSurface(ForetGeom.geom),
                                WGS84WM))) \
                    .filter(ForetGeom.id_nat_frt == user_search.upper()) \
                    .first()
            elif REG_COMMUNE_CODE.match(user_search):
                session = get_user_session(INSEE_SESSION)
                geo_marker = session.query(
                        func.ST_AsEWKT(
                            func.ST_Transform(
                                func.ST_Centroid(CommuneGeom.geom),
                                WGS84WM))) \
                    .filter(CommuneGeom.code == user_search) \
                    .first()
        if geo_marker is not None:
            renderer_settings = settings.get_renderer().renderer_settings
            params.append(ForetGeom.geom_3857.ST_DWithin(
                geo_marker, getattr(renderer_settings, 'search_distance', 25) * 1000))
        elif user_search:
            params.append(or_(
                func.lower(Foret.libelle_recherche).contains(user_search),
                func.lower(InformationForet.libelle).contains(user_search),
                func.lower(ForetSurCommune.libelle_insee).contains(user_search)
            ))

        catalog = get_utility(ICatalog)
        if geo_marker is not None:
            query = session.query(Foret,
                                  InformationForet,
                                  func.ST_Distance(
                                      ForetGeom.geom_3857,
                                      geo_marker).label('distance'))
        else:
            query = session.query(Foret,
                                  InformationForet,
                                  literal_column('0').label('distance'))
        query = query \
            .join(ProprietaireForet, Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
            .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .outerjoin(ForetSurCommune, Foret.id_nat_frt == ForetSurCommune.id_nat_frt) \
            .outerjoin(ForetGeom, Foret.id_nat_frt == ForetGeom.id_nat_frt) \
            .filter(and_(*params)) \
            .distinct()
        if geo_marker is not None:
            query = query.order_by('distance')
        else:
            query = query.order_by(Foret.libelle_recherche)

        results_ids = set()
        for foret, info, distance in query:
            index = -1
            if foret.id_nat_frt not in results_ids:
                for index, item in enumerate(CatalogResultSet(CatalogQuery(catalog).query(
                    And(Eq(catalog['content_type'], FOREST_CONTENT_TYPE),
                        Eq(catalog['forests'], foret.id_nat_frt),
                        Any(catalog['workflow_state'], VISIBLE_STATES))))):
                    if (item not in results) and (foret.id_nat_frt not in results_ids):
                        results.append((item, distance))
                        results_ids.add(foret.id_nat_frt)
                else:
                    if index < 0:
                        results.append((FakeForest(foret.id_nat_frt), distance))
                        results_ids.add(foret.id_nat_frt)
        count = len(results)
        return (results[start:start+length], count, aggregations) if get_count else results


@adapter_config(name='activity',
                context=(IForestHubSearchFolder, IPyAMSLayer),
                provides=IForestHubSearchFolderQuery)
class ForestHubSearchFolderActivitySearchAdapter(ContextRequestAdapter):
    """Forest hub search folder activities search adapter"""

    def get_results(self, context, sort_index=None, reverse=None, limit=None,
                    start=0, length=None, ignore_cache=False, get_count=False, request=None,
                    aggregates=None, settings=None):
        return ()
