#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""
from uuid import uuid4

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Eq
from pyramid.decorator import reify
from pyramid.httpexceptions import HTTPNotFound
from pyramid.renderers import render_to_response
from pyramid.response import Response
from pyramid.view import view_config
from z3c.form.button import Button, Buttons, handler
from z3c.form.field import Fields
from z3c.form.interfaces import DISPLAY_MODE
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface, alsoProvides, implementer
from zope.lifecycleevent import ObjectCreatedEvent
from zope.location import locate

from onf_website.reference.forest.schema import ForestField
from onf_website.shared.forest import FOREST_CONTENT_TYPE, FakeForest, IForestManager, IWfForest
from onf_website.shared.forest.interfaces import IWfForestInfo, MANAGE_FOREST_PERMISSION
from onf_website.shared.forest.zmi.interfaces import IDraftGalleryContentsView, \
    IPublishedGalleryContentsView
from onf_website.skin.zmi import onf_website
from pyams_catalog.query import CatalogResultSet
from pyams_content.component.gallery import IGallery
from pyams_content.component.gallery.zmi import GalleryMediasView
from pyams_content.component.gallery.zmi.file import GalleryMediaAddForm, GalleryMediaAddMenu
from pyams_content.component.gallery.zmi.paragraph import GalleryContentsView
from pyams_content.interfaces import COMMENT_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common import IWfSharedContentRoles
from pyams_content.workflow import DRAFT, PROPOSED
from pyams_file.interfaces import IThumbnails
from pyams_form.form import AJAXAddForm, InnerEditForm, ajax_config
from pyams_form.help import FormHelp, HelpContentProvider
from pyams_form.interfaces.form import IFormContextPermissionChecker, IWidgetForm, \
    IWidgetsPrefixViewletsManager, \
    IWidgetsSuffixViewletsManager
from pyams_form.schema import ResetButton
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IInnerPage
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.skin import apply_skin
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url, generate_url
from pyams_viewlet.viewlet import EmptyContentProvider, Viewlet, viewlet_config
from pyams_workflow.interfaces import IWorkflowInfo, IWorkflowState, IWorkflowVersions
from pyams_zmi.form import AdminAddForm, AdminEditForm
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.skin import AdminSkin


__docformat__ = 'restructuredtext'

from onf_website import _


class IForestUserGalleryInputInnerForm(Interface):
    """Forest user gallery input form marker interface"""


class IWfForestUserGalleryInput(Interface):
    """User forest gallery input form interface"""

    forest_id = ForestField(title=_("Forest national ID"),
                            description=_("You can select a forest to display it's medias "
                                          "gallery"),
                            required=True)


class IWfForestUserGalleryInputButtons(Interface):
    """User forest gallery input form buttons interface"""

    reset = ResetButton(name='reset', title=_("Reset"))
    publish = Button(name='publish', title=_("Save and request publication"))
    save = Button(name='save', title=_("Save my proposition"))


@pagelet_config(name='forest-user-gallery.html',
                context=IForestManager, layer=IPyAMSLayer,
                permission=MANAGE_FOREST_PERMISSION)
@ajax_config(name='forest-user-gallery.json',
             context=IForestManager, layer=IPyAMSLayer,
             permission=MANAGE_FOREST_PERMISSION,
             base=AJAXAddForm)
@implementer(IWidgetForm, IInnerPage, IObjectData)
class ForestUserGalleryInputForm(AdminAddForm):
    """Forest user gallery input form"""

    legend = _("What is the forest for which you would like to provide pictures?")
    prefix = 'form.'

    fields = Fields(IWfForestUserGalleryInput)

    buttons = Buttons(IWfForestUserGalleryInputButtons)
    handleActions = True

    edit_permission = MANAGE_FOREST_PERMISSION

    object_data = {
        'ams-plugins': 'onf_website',
        'ams-plugin-onf_website-src': get_resource_path(onf_website),
        'ams-reset-handler': 'ONF_website.forest.resetGalleryForm'
    }

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        forest_id = self.widgets.get('forest_id')
        if forest_id is not None:
            forest_id.object_data = {
                'ams-change-handler': 'MyAMS.helpers.select2ChangeHelper',
                'ams-change-stop-propagation': 'true',
                'ams-select2-change-on-reset': 'false',
                'ams-select2-helper-type': 'html',
                'ams-select2-helper-url': absolute_url(self.context, self.request,
                                                       'get-forest-user-gallery.html'),
                'ams-select2-helper-argument': 'forest_id',
                'ams-select2-helper-target': '#forest-user-gallery-helper'
            }
            alsoProvides(forest_id, IObjectData)

    def updateActions(self):
        super().updateActions()
        publish = self.actions.get('publish')
        if publish is not None:
            publish.addClass('btn-primary')
        save = self.actions.get('save')
        if save is not None:
            save.addClass('btn-primary')

    @handler(buttons['reset'])
    def handleReset(self, action):
        return

    @handler(buttons['publish'])
    def handlePublish(self, action):
        forest = self.handleSave(self, action)
        state = IWorkflowState(forest)
        if state.state != PROPOSED:
            info = IWorkflowInfo(forest)
            contributors = forest.contributors
            forest.contributors = contributors | {self.request.principal.id}
            info.fire_transition_toward(PROPOSED, check_security=False)
            dc = IZopeDublinCore(forest)
            forest.baseline_timestamp = dc.modified
        return forest

    @handler(buttons['save'])
    def handleSave(self, action):
        data, errors = self.extractData()
        forest_id = data.get('forest_id')
        if forest_id is None:
            raise HTTPNotFound
        catalog = get_utility(ICatalog)
        params = Eq(catalog['content_type'], FOREST_CONTENT_TYPE) & \
            Eq(catalog['forests'], forest_id)
        has_results, results = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
        if not has_results:
            raise HTTPNotFound
        forest = next(results)
        versions = IWorkflowVersions(forest)
        if versions.has_version({DRAFT, PROPOSED}):
            forest = versions.get_versions({DRAFT, PROPOSED}, sort=True)[-1]
        else:
            version = versions.get_last_versions()[-1]
            info = IWorkflowInfo(version)
            translate = self.request.localizer.translate
            forest = info.fire_transition_toward(DRAFT, check_security=False,
                                                 comment=translate(_("New draft created from "
                                                                     "gallery presentation module")))
        forest.gallery_comment = self.request.params.get('form.widgets.gallery_comment')
        return forest

    def add(self, object):
        pass

    def nextURL(self):
        pass

    def get_ajax_output(self, changes):
        translate = self.request.localizer.translate
        message = None
        for action in self.actions.executedActions:
            if action.field is IWfForestUserGalleryInputButtons['save']:
                message = translate(_("Your proposal has been saved. You didn't requested its "
                                      "publication, but you can find it to complete or update "
                                      "just by using this form and selecting your forest "
                                      "again..."))
            elif action.field is IWfForestUserGalleryInputButtons['publish']:
                message = translate(_("Your proposed images have been saved and the editorial "
                                      "team has been notified!"))
        result = {'status': 'reload'}
        if message:
            result['messagebox'] = {
                'status': 'success',
                'title': translate(_("Input saved!")),
                'content': message,
                'icon': 'fa fa-fw fa-user',
                'timeout': 10000
            }
        return result


@viewlet_config(name='forest-user-gallery',
                context=IForestManager, layer=IAdminLayer, view=ForestUserGalleryInputForm,
                manager=IWidgetsSuffixViewletsManager, weight=10)
class ForestUserGalleryInputFormSuffix(Viewlet):
    """Forest user input form suffix"""

    def render(self):
        return '<div id="forest-user-gallery-helper"></div>' \
               '<div id="forest-user-gallery-feedback"></div>'


class ForestUserGalleryHelp(HelpContentProvider):
    """Forest user gallery view"""

    def update(self):
        manager = get_utility(IForestManager)
        if manager.gallery_help:
            help = self.help = FormHelp(self.context, self.request, self.view)
            help.message = manager.gallery_help
            help.message_format = 'rest'


class BaseForestUserGalleryView(GalleryContentsView):
    """Base user forest gallery view"""

    gallery_intf = IGallery
    gallery_name = ''

    @reify
    def gallery(self):
        registry = self.request.registry
        return registry.queryAdapter(self.context, self.gallery_intf, self.gallery_name)


@template_config(template='templates/gallery-medias-draft.pt', layer=IAdminLayer)
@implementer(IDraftGalleryContentsView)
class ForestUserDraftGalleryView(BaseForestUserGalleryView):
    """Forest user draft gallery edit form"""

    _legend = _("Gallery of proposed medias")
    gallery_name = 'draft'

    permission = MANAGE_FOREST_PERMISSION


class ForestUserGalleryCommentsEditForm(InnerEditForm):
    """Forest user gallery comments edit form"""

    legend = _("Publication comments")
    fields = Fields(IWfForestInfo).select('gallery_comment')

    label_css_class = 'hidden'
    input_css_class = 'col-md-12'

    edit_permission = MANAGE_FOREST_PERMISSION


@adapter_config(context=(IWfForest, IAdminLayer, ForestUserGalleryCommentsEditForm),
                provides=IFormContextPermissionChecker)
class ForestUserGalleryCommentsEditFormPermissionChecker(ContextRequestViewAdapter):
    """Forest user gallery comment edit form permission checker"""

    edit_permission = MANAGE_FOREST_PERMISSION


@template_config(template='templates/gallery-medias-published.pt', layer=IAdminLayer)
@implementer(IPublishedGalleryContentsView)
class ForestUserPublishedGalleryView(BaseForestUserGalleryView):
    """Forest user published gallery edit form"""

    _legend = _("Gallery of selected medias")


class ForestUserGalleriesView(EmptyContentProvider):
    """User forest galleries view"""

    def __init__(self, context, request):
        super().__init__(context, request)
        self.views = [
            ForestUserGalleryHelp(context, request, self),
            ForestUserDraftGalleryView(context, request, self),
            ForestUserGalleryCommentsEditForm(context, request),
            ForestUserPublishedGalleryView(context, request, self)
        ]

    def update(self):
        super().update()
        [view.update() for view in self.views]

    def render(self, template_name=''):
        return ''.join((
            view.render()
            for view in self.views
        ))


@view_config(name='get-forest-user-gallery.html',
             context=IForestManager, request_type=IPyAMSLayer,
             permission=MANAGE_FOREST_PERMISSION, xhr=True)
def get_forest_user_gallery(request):
    """Forest user gallery input form"""
    apply_skin(request, AdminSkin)
    forest_id = request.params.get('forest_id')
    if forest_id is None:
        raise HTTPNotFound
    catalog = get_utility(ICatalog)
    params = Eq(catalog['content_type'], FOREST_CONTENT_TYPE) & \
        Eq(catalog['forests'], forest_id)
    has_results, results = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
    if not has_results:
        fake = FakeForest()
        fake.forest_ids = [forest_id]
        if not fake.forest_refs:
            raise HTTPNotFound
        manager = request.context
        content = manager.shared_content_factory()
        request.registry.notify(ObjectCreatedEvent(content))
        uuid = str(uuid4())
        manager[uuid] = content
        forest = manager.shared_content_factory.content_class()
        locate(forest, content)
        request.registry.notify(ObjectCreatedEvent(forest))
        forest.title = {'fr': fake.title}
        forest.short_name = forest.title.copy()
        forest.content_url = generate_url(fake.title)
        forest.forest_ids = fake.forest_ids
        forest.creator = request.principal.id
        owner = request.principal.id
        if owner in manager.contributors:
            forest.owner = owner
        else:
            forest.owner = manager.default_owner
        if not request.has_permission(COMMENT_CONTENT_PERMISSION, context=forest):
            IWfSharedContentRoles(forest).readers = {request.principal.id}
        IWorkflowVersions(content).add_version(forest, None)
        translate = request.localizer.translate
        IWorkflowInfo(forest).fire_transition('init',
                                              comment=translate(_("Draft created from gallery "
                                                                  "presentation module")))
    else:
        forest = next(results)
        versions = IWorkflowVersions(forest)
        if versions.has_version({DRAFT, PROPOSED}):
            forest = versions.get_versions({DRAFT, PROPOSED}, sort=True)[-1]
        else:
            version = versions.get_last_versions()[-1]
            info = IWorkflowInfo(version)
            contributors = (version.contributors or set()).copy()
            version.contributors = contributors | {request.principal.id}
            translate = request.localizer.translate
            forest = info.fire_transition_toward(DRAFT, check_security=False,
                                                 comment=translate(_("New draft created from "
                                                                     "gallery presentation module")))
            forest.contributors = version.contributors = contributors
    form = ForestUserGalleriesView(forest, request)
    form.update()
    return Response(form.render())


@viewlet_config(name='add-media.menu',
                context=IWfForest, view=IDraftGalleryContentsView,
                manager=IWidgetTitleViewletManager)
class ForestUserDraftGalleryMediaAddMenu(GalleryMediaAddMenu):
    """Forest user draft gallery media add menu"""

    url = 'add-draft-media.html'

    permission = MANAGE_FOREST_PERMISSION

    def get_url(self):
        gallery = self.__parent__.gallery
        return absolute_url(gallery, self.request, self.url)


@pagelet_config(name='add-draft-media.html',
                context=IGallery, layer=IPyAMSLayer,
                permission=MANAGE_FOREST_PERMISSION)
@ajax_config(name='add-draft-media.json',
             context=IGallery, layer=IPyAMSLayer, base=AJAXAddForm)
class ForestUserDraftGalleryMediaAddForm(GalleryMediaAddForm):
    """Forest user draft gallery media add form"""

    edit_permission = MANAGE_FOREST_PERMISSION

    def get_ajax_output(self, changes):
        forest = get_parent(self.context, IWfForest)
        manager = get_parent(forest, IForestManager)
        return {
            'status': 'reload',
            'location': absolute_url(manager, self.request, 'get-forest-user-gallery.html',
                                     query={'forest_id': forest.forest_ids[0]}),
            'target': '#forest-user-gallery-helper'
        }


#
# Forest presentation gallery management
#

@pagelet_config(name='gallery.html',
                context=IWfForest, layer=IPyAMSLayer,
                permission=MANAGE_FOREST_PERMISSION)
@implementer(IWidgetForm, IInnerPage)
class ForestManagerGalleryView(AdminEditForm):
    """Forest manager gallery view """

    legend = _("Update presentation gallery")
    fields = Fields(Interface)
    buttons = Buttons(Interface)


@viewlet_config(name='gallery-medias-draft',
                context=IWfForest, view=ForestManagerGalleryView,
                manager=IWidgetsPrefixViewletsManager, weight=10)
@template_config(template='templates/gallery-manager-draft.pt', layer=IPyAMSLayer)
class ForestManagerDraftGalleryView(ForestUserDraftGalleryView):
    """Forest manager draft gallery view"""

    def __init__(self, context, request, view, manager):
        super().__init__(context, request, view)


@viewlet_config(name='gallery-comments',
                context=IWfForest, view=ForestManagerGalleryView,
                manager=IWidgetsPrefixViewletsManager, weight=20)
class ForestManagerGalleryCommentsView(Viewlet):
    """Forest manager gallery comments view"""

    def __new__(cls, context, request, view, manager):
        if not context.gallery_comment:
            return None
        return EmptyContentProvider.__new__(cls)

    def __init__(self, context, request, view, manager):
        super().__init__(context, request, view, manager)
        self.comment_form = ForestUserGalleryCommentsEditForm(context, request)
        self.comment_form.mode = DISPLAY_MODE

    def update(self):
        super().update()
        self.comment_form.update()

    def render(self):
        return self.comment_form.render()


@viewlet_config(name='gallery-medias-published',
                context=IWfForest, view=ForestManagerGalleryView,
                manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/gallery-manager-published.pt', layer=IPyAMSLayer)
class ForestManagerPublishedGalleryView(ForestUserPublishedGalleryView):
    """Forest manager published gallery view"""

    def __init__(self, context, request, view=None, manager=None):
        super().__init__(context, request, view)


@viewlet_config(name='add-media.menu',
                context=IWfForest, view=IPublishedGalleryContentsView,
                manager=IWidgetTitleViewletManager)
class PublishedGalleryMediaAddMenu(GalleryMediaAddMenu):
    """Published gallery media add menu"""

    url = 'add-published-media.html'

    def get_url(self):
        gallery = self.__parent__.gallery
        return absolute_url(gallery, self.request, self.url)


@pagelet_config(name='add-published-media.html',
                context=IGallery, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-published-media.json',
             context=IGallery, layer=IPyAMSLayer, base=AJAXAddForm)
class ForestPublishedGalleryMediaAddForm(GalleryMediaAddForm):
    """Published gallery media add form"""

    def get_ajax_output(self, changes):
        return {
            'status': 'reload',
            'location': absolute_url(self.context, self.request, 'get-published-medias.html'),
            'target': '#gallery_medias_published'
        }


@view_config(name='get-published-medias.html',
             context=IGallery, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
class PublishedMediasView(GalleryMediasView):
    """Published gallery medias view"""

    @property
    def legend(self):
        return self.request.localizer.translate(_("Gallery of selected medias"))

    def __call__(self):
        forest = get_parent(self.context, IWfForest)
        return render_to_response('templates/gallery-manager-published-inner.pt', {
            'context': forest,
            'view': self
        }, request=self.request)

    @property
    def gallery(self):
        return self.context


@view_config(name='copy-from-draft.json',
             context=IGallery, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def copy_media_from_draft(request):
    """Copy media from draft gallery to published"""
    translate = request.localizer.translate
    target = request.context
    forest = get_parent(target, IWfForest)
    source = request.registry.queryAdapter(forest, IGallery, name='draft')
    if source is None:
        return {
            'status': 'error',
            'message': translate(_("Can't find source gallery!"))
        }
    media_name = request.params.get('source')
    if media_name is None:
        return {
            'status': 'error',
            'message': translate(_("Missing source element name!"))
        }
    media = source.get(media_name)
    if media is None:
        return {
            'status': 'error',
            'message': translate(_("Can't find provided element in source gallery!"))
        }
    del source[media_name]
    target.append(media)
    thumbnails = IThumbnails(media.data)
    return {
        'status': 'success',
        'result': {
            'name': media.__name__,
            'location': absolute_url(media, request=request),
            'src': absolute_url(thumbnails.get_thumbnail('128x128'), request=request),
            'href': absolute_url(thumbnails.get_thumbnail('800x600'), request=request)
        }
    }
