#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""ONF_website.shared.forest.interfaces module

"""

from zope.interface import Interface
from zope.schema import Choice, Datetime, Dict, Int, Text, TextLine

from onf_website.reference.forest.interfaces import FOREST_OWNERS_CATEGORIES
from onf_website.reference.forest.schema import ForestsListField
from pyams_content.component.gallery import IGalleryTarget
from pyams_content.component.illustration import ILLUSTRATION_KEY
from pyams_content.shared.common import ISharedContent
from pyams_content.shared.common.interfaces import ISharedToolPortalContext, \
    IWfSharedContentPortalContext
from pyams_i18n.schema import I18nHTMLField
from pyams_portal.interfaces import PORTAL_TEMPLATES_VOCABULARY
from pyams_security.schema import Principal
from pyams_sequence.interfaces import IInternalReferencesList
from pyams_sequence.schema import InternalReferenceField


__docformat__ = 'restructuredtext'

from onf_website import _


FOREST_CONTENT_TYPE = 'forest'
FOREST_CONTENT_NAME = _("Forest")


MANAGE_FOREST_PERMISSION = 'onf.ManageForest'
'''Permission required to manage a forest content'''


class IForestManagerInfo(Interface):
    """Forest manager base interface"""

    baseline_help = Text(title=_("Baseline help"),
                         description=_("This text will be displayed in user's baseline input "
                                       "form; you can use reStructuredText to include links or "
                                       "CSS classes"),
                         required=False)

    baseline_max_length = Int(title=_("Baseline max length"),
                              description=_("Maximum baseline length; this length is only "
                                            "informative and not blocking"),
                              required=True,
                              default=2000)

    gallery_help = Text(title=_("Gallery help"),
                        description=_("This text will be displayed in user's gallery input form; "
                                      "you can use reStructuredText to include links or CSS "
                                      "classes"),
                        required=False)

    default_owner = Principal(title=_("Default owner"),
                              description=_("This principal will be set as owner of all forests "
                                            "created automatically, except if created by a known "
                                            "contributor"))

    default_hearing_source = Choice(title=_("Default origin site"),
                                    description=_("This site will be used as source site for all "
                                                  "forests contents"),
                                    required=False,
                                    vocabulary='PyAMS shared sites')

    default_source_folder = InternalReferenceField(
        title=_("Default source folder"),
        description=_("Main folder to which content is attached; if this reference "
                      "is set, canonical URL will be based on this object's URL for all forests"),
        required=False)

    def get_default_source(self):
        """Get default hearing source"""

    templates = Dict(title=_("Custom templates"),
                     description=_("You can define specific templates for each forest owner "
                                   "category"),
                     key_type=Choice(vocabulary=FOREST_OWNERS_CATEGORIES),
                     value_type=Choice(vocabulary=PORTAL_TEMPLATES_VOCABULARY,
                                       required=False),
                     required=False)


DOM_ILLUSTRATION_KEY = '{0}::link'.format(ILLUSTRATION_KEY)


class IForestManager(ISharedToolPortalContext, IForestManagerInfo):
    """Forest manager interface"""


class IForestManagerFactory(Interface):
    """Forest manager factory interface"""


class IWfForestInfo(Interface):
    """Forest information interface"""

    forest_ids = ForestsListField(title=_("Forests national IDs"),
                                  description=_("You can select one or several forests which are "
                                                "matching this forest content; content location "
                                                "will be based automatically on this selection"),
                                  required=True)

    baseline_draft = I18nHTMLField(title=_("Proposed baseline"),
                                   description=_("Proposed baseline text for this forest"),
                                   required=False)

    baseline_principal = Principal(title=_("Last proposal"),
                                   description=_("Principal who proposed this baseline"),
                                   required=False)

    baseline_principal_status = TextLine(title=_("Last proposal"),
                                         readonly=True)

    baseline_timestamp = Datetime(title=_("Proposal timestamp"),
                                  description=_("Timestamp of this baseline proposal"),
                                  required=False)

    baseline_published = I18nHTMLField(title=_("Published baseline"),
                                       description=_("Published baseline text for this forest"),
                                       required=False)

    gallery_comment = Text(title=_("Gallery comment"),
                           description=_("This comment is only for internal use, if required"),
                           required=False)

    def get_forest_refs(self):
        """Get DB records matching forest IDs"""


class IWfForest(IWfSharedContentPortalContext, IWfForestInfo, IInternalReferencesList):
    """Forest interface"""


class IWfRealForest(Interface):
    """Real forest marker interface"""


class IWfForestFactory(Interface):
    """Forest interface"""


class IForest(ISharedContent):
    """Workflow managed forest content interface"""
