#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq
from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent import IObjectAddedEvent

from onf_website.shared.contact import Contact, IContactAssignmentsContainer
from onf_website.shared.contact.interfaces import CONTACT_CONTENT_TYPE, IContactManager, \
    IContactManagerFactory
from onf_website.shared.contact.interfaces.theme import IContactTheme, IContactThemeManager, \
    IContactThemeManagerTarget
from pyams_catalog.query import CatalogResultSet
from pyams_content.shared.common.interfaces import ISharedContentFactory
from pyams_content.shared.common.manager import SharedTool
from pyams_utils.adapter import adapter_config
from pyams_utils.list import boolean_iter, unique_iter
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_workflow.interfaces import IWorkflow


__docformat__ = 'restructuredtext'


@implementer(IContactManager, IContactThemeManagerTarget)
class ContactManager(SharedTool):
    """Contact manager class"""

    shared_content_type = CONTACT_CONTENT_TYPE
    shared_content_menu = False

    def get_contacts(self, theme):
        catalog = get_utility(ICatalog)
        params = Eq(catalog['content_type'], self.shared_content_type) & \
            Any(catalog['workflow_state'], IWorkflow(self).published_states)
        yield from filter(lambda x: x[1] is not None,
                          map(lambda x: (
                              x, IContactAssignmentsContainer(x).get_active_assignments(theme)),
                              CatalogResultSet(CatalogQuery(catalog).query(params))))

    def get_contacts_matching(self, theme, insee_code=None, dept_code=None):
        """Get iterator over contacts matching given theme and postal code"""
        catalog = get_utility(ICatalog)
        params = Eq(catalog['content_type'], self.shared_content_type) & \
            Any(catalog['workflow_state'], IWorkflow(self).published_states)
        has_items, items = boolean_iter(unique_iter(
            filter(lambda x: x[1] is not None,
                   map(lambda x: (
                       x, IContactAssignmentsContainer(x).get_matching_assignment(theme,
                                                                                  insee_code,
                                                                                  dept_code)),
                       CatalogResultSet(CatalogQuery(catalog).query(params))))))
        if has_items:
            yield from items
        else:
            if not IContactTheme.providedBy(theme):
                themes = IContactThemeManager(self, None)
                if themes is not None:
                    theme = themes.get(theme)
            if theme is not None:
                contact = theme.get_target()
                if contact is not None:
                    yield contact, None


@utility_config(provides=IContactManagerFactory)
class ContactManagerFactory(object):
    """Default contact manager factory"""

    def __new__(cls):
        return ContactManager


@adapter_config(context=IContactManager, provides=ISharedContentFactory)
def contact_manager_content_factory(context):
    """Contact manager content factory"""
    return Contact


@subscriber(IObjectAddedEvent, context_selector=IContactManager)
def handle_added_contact_manager(event):
    """Register contact manager when added"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IContactManager)
