#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from datetime import datetime

from persistent import Persistent
from pyramid_mailer import IMailer
from sqlalchemy import Column, DateTime, Integer, JSON, Sequence, Unicode
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from onf_website.shared.contact import IContactManager
from onf_website.shared.contact.interfaces.handler import CONTACT_FORM_HANDLER_ANNOTATIONS_KEY, \
    IContactFormHandlerInfo, IContactFormHandlerTarget, ILeadLog
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin
from pyams_content.shared.form import IFormFieldContainer, IFormHandler
from pyams_i18n.interfaces import II18n
from pyams_mail.message import HTMLMessage
from pyams_security.interfaces import ISecurityManager
from pyams_security.interfaces.notification import INotificationSettings
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_utility, query_utility, utility_config
from pyams_utils.request import query_request
from pyams_workflow.interfaces import IWorkflowState


__docformat__ = 'restructuredtext'

from onf_website import _


@factory_config(IContactFormHandlerInfo)
class ContactFormHandlerInfo(Persistent):
    """Contact form handler persistent info"""

    service_name = FieldProperty(IContactFormHandlerInfo['service_name'])
    crm_source = FieldProperty(IContactFormHandlerInfo['crm_source'])
    source_address = FieldProperty(IContactFormHandlerInfo['source_address'])
    source_name = FieldProperty(IContactFormHandlerInfo['source_name'])
    target_address = FieldProperty(IContactFormHandlerInfo['target_address'])
    target_name = FieldProperty(IContactFormHandlerInfo['target_name'])
    notification_message = FieldProperty(IContactFormHandlerInfo['notification_message'])
    confirm_message = FieldProperty(IContactFormHandlerInfo['confirm_message'])
    redirection_message = FieldProperty(IContactFormHandlerInfo['redirection_message'])


@adapter_config(context=IContactFormHandlerTarget, provides=IContactFormHandlerInfo)
def contact_form_handler_factory(context):
    """Contact form handler information factory"""
    return get_annotation_adapter(context, CONTACT_FORM_HANDLER_ANNOTATIONS_KEY,
                                  IContactFormHandlerInfo)


PARENT_SESSION = 'INTERNET'
PARENT_SCHEMA = 'leads'

LEADS_SEQUENCE = Sequence('leads.log_id_seq', quote=False)


@implementer(ILeadLog)
class LeadLog(DynamicSchemaMixin, Base):
    """Leads log table schema"""

    __tablename__ = 'log'
    __schema__ = PARENT_SCHEMA

    id = Column(Integer, LEADS_SEQUENCE, primary_key=True)
    form_oid = Column(Unicode(12))
    form_version = Column(Integer)
    submit_date = Column(DateTime, default=datetime.utcnow)
    submit_email = Column(Unicode(255))
    submit_data = Column(JSON)
    rgpd_consent = Column(Unicode)
    contacts = Column(Unicode)

    @staticmethod
    def get_oid(form, id):
        sequence = ISequentialIdInfo(form)
        return '{oid}-{seq:0>6}'.format(oid=sequence.get_short_oid().replace(' ', '-'),
                                        seq=id)

    @property
    def oid(self):
        return '{oid}-{seq:0>6}'.format(oid=self.form_oid.replace(' ', '-'),
                                        seq=self.id)


@utility_config(name='contact', provides=IFormHandler)
class ContactFormHandler(object):
    """Contact form handler"""

    label = _("Contact form handler")
    weight = 20

    target_interface = IContactFormHandlerTarget
    handler_info = IContactFormHandlerInfo

    @staticmethod
    def build_message(template, data):
        if template:
            return template.format(**data)
        else:
            return '<br />'.join(('{}: {}'.format(k, v or '--') for k, v in data.items()))

    def handle(self, form, data, user_data):
        contacts = query_utility(IContactManager)
        if contacts is None:
            return
        # check theme
        fields = IFormFieldContainer(form)
        has_themes, themes_field = boolean_iter(fields.find_fields('contact_theme'))
        if not has_themes:
            has_themes, themes_field = boolean_iter(fields.find_fields('contact_themes_list'))
            if not has_themes:
                return
        themes = data.get(next(themes_field).name)
        if not isinstance(themes, (list, tuple, set)):
            themes = [themes]
        # check department
        dept_code = None
        has_dept, dept_code_field = boolean_iter(fields.find_fields('department'))
        if has_dept:
            dept_code = data.get(next(dept_code_field).name)
        # check INSEE code
        insee_code = None
        has_code, insee_code_field = boolean_iter(fields.find_fields('postal_code'))
        if has_code:
            insee_code = data.get(next(insee_code_field).name)
        # send notification messages
        security = get_utility(ISecurityManager)
        settings = INotificationSettings(security)
        if settings.enable_notifications:
            mailer = query_utility(IMailer, name=settings.mailer)
            if mailer is not None:
                request = query_request()
                i18n_form = II18n(form)
                handler_info = self.handler_info(form)
                i18n_handler = II18n(handler_info)
                # build notification message
                session = get_user_session(PARENT_SESSION)
                log_id = session.execute(LEADS_SEQUENCE)
                user_data['_reference'] = LeadLog.get_oid(form, log_id)
                body = self.build_message(
                    i18n_handler.query_attribute('notification_message', request=request),
                    user_data)
                # extract default contact
                notified = []
                targets = []
                default_email = handler_info.target_address
                if default_email:
                    default_email = default_email.split(';')
                    for mail in default_email:
                        target = '{} <{}>'.format(handler_info.target_name, mail) \
                            if handler_info.target_name else mail
                        message = HTMLMessage(
                            subject='[{}] {}'.format(settings.subject_prefix,
                                                     i18n_handler.query_attribute('service_name',
                                                                                  request=request)),
                            fromaddr='{} <{}>'.format(handler_info.source_name,
                                                      handler_info.source_address),
                            toaddr=target,
                            html=body)
                        mailer.send(message)
                        notified.append(target)
                # extract other contacts list
                for theme in themes:
                    has_items, items = boolean_iter(
                        contacts.get_contacts_matching(theme, insee_code, dept_code))
                    if has_items:
                        for contact, assignment in items:
                            url, label = None, None
                            # if assignment or contact is given an URL, the user is redirected
                            # to this URL and no mail is sent to this contact...
                            if assignment is not None:
                                url, label = assignment.target_url, \
                                             II18n(assignment).query_attribute('target_url_label',
                                                                               request=request)
                            if (not url) and (not assignment.mail_address):
                                url = contact.target_url
                                label = II18n(contact).query_attribute('target_url_label',
                                                                       request=request)
                            if url:
                                targets.append((url, label or None))
                                continue
                            name = contact.contact_name
                            mail = (assignment.mail_address if assignment is not None else None) or \
                                contact.mail_address
                            if mail and ((not default_email) or (mail not in default_email)):
                                target = '{} <{}>'.format(name, mail) if name else mail
                                message = HTMLMessage(
                                    subject='[{}] {}'.format(settings.subject_prefix,
                                                             i18n_handler.query_attribute(
                                                                 'service_name', request=request)),
                                    fromaddr='{} <{}>'.format(handler_info.source_name,
                                                              handler_info.source_address),
                                    toaddr=target,
                                    html=body)
                                mailer.send(message)
                                notified.append(target)
                # if contacts were notified, log lead and send confirmation message
                if notified:
                    email_address = request.localizer.translate(_("no address provided"))
                    has_email, email_field = boolean_iter(fields.find_fields('mail'))
                    if has_email:
                        email_address = data.get(next(email_field).name)
                    # log new lead
                    log = LeadLog(
                        id=log_id,
                        form_oid=ISequentialIdInfo(form).get_short_oid(),
                        form_version=IWorkflowState(form).version_id,
                        submit_email=email_address,
                        submit_data=user_data,
                        contacts=', '.join(notified)
                    )
                    if form.rgpd_consent:
                        log.rgpd_consent = i18n_form.query_attribute('rgpd_warning',
                                                                     request=request)
                    session.add(log)
                    # send confirmation message
                    if has_email and email_address:
                        confirm_message = i18n_handler.query_attribute('confirm_message',
                                                                       request=request)
                        if confirm_message:
                            message = HTMLMessage(
                                subject='[{}] {}'.format(settings.subject_prefix,
                                                         II18n(handler_info).query_attribute(
                                                             'service_name', request=request)),
                                fromaddr='{} <{}>'.format(handler_info.source_name,
                                                          handler_info.source_address),
                                toaddr=email_address,
                                html=confirm_message.format(**user_data))
                            mailer.send(message)
                # if target URLs where found, send output links to user
                if targets:
                    message = i18n_handler.query_attribute('redirection_message',
                                                           request=request)
                    if message:
                        links = '<ul>{}</ul>'.format(
                            ''.join(('<li><a href="{href}" target="_blank">{label}</a></li>'.format(
                                href=target[0], label=target[1] or target[0])
                                for target in targets)))
                        return message.format(links=links)
