#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.interface import implementer
from zope.location import Location
from zope.schema import Choice, List
from zope.schema.fieldproperty import FieldProperty
from zope.schema.interfaces import IChoice, IList
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from onf_website.shared.contact import IContactManager
from onf_website.shared.contact.interfaces.field import \
    CONTACT_THEMES_LIST_FIELD_VALUES_VOCABULARY, CONTACT_THEME_FIELD_INFO_ANNOTATION_KEY, \
    CONTACT_THEME_FIELD_VALUES_VOCABULARY, IContactThemeFieldInfo, \
    CONTACT_FORM_THEMES_FIELD_VALUES_VOCABULARY
from onf_website.shared.contact.interfaces.form import IInternalFormLinkTheme
from onf_website.shared.contact.interfaces.theme import IContactThemeManager
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_content.component.links import IInternalLink
from pyams_content.shared.form import IWfForm, IFormFieldContainer
from pyams_content.shared.form.field import ChoiceFieldFactory, ListFieldFactory
from pyams_content.shared.form.interfaces import IFormField, IFormFieldDataConverter, \
    IFormFieldFactory
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_utils.adapter import ContextRequestAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.list import boolean_iter
from pyams_utils.registry import query_utility, utility_config
from pyams_utils.request import query_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'

from onf_website import _


#
# Simple contact theme field
#

class IContactThemeField(IChoice):
    """Contact theme field interface"""


@implementer(IContactThemeField)
class ContactThemeField(Choice):
    """Contact theme field"""


@utility_config(name='contact_theme', provides=IFormFieldFactory)
class ContactThemeFormFieldFactory(ChoiceFieldFactory):
    """Contact theme form field factory"""

    label = _("Contact theme")
    weight = 101

    field_factory = ContactThemeField

    def get_schema_field(self, field):
        i18n = II18n(field)
        result = self.field_factory(title=i18n.query_attribute('label'),
                                    description=i18n.query_attribute('description'),
                                    required=field.required,
                                    default=i18n.query_attribute('default'),
                                    vocabulary=ContactThemeFieldValuesVocabulary(field))
        result.__name__ = field.name
        return result


#
# Multiple contact themes field
#

class IContactThemesListField(IList):
    """Contact themes list field interface"""


@implementer(IContactThemesListField)
class ContactThemesListField(List):
    """Contact themes list field"""


@utility_config(name='contact_themes_list', provides=IFormFieldFactory)
class ContactThemesListFormFieldFactory(ListFieldFactory):
    """Contact themes list form field factory"""

    label = _("Contact themes list")
    weight = 102

    field_factory = ContactThemesListField

    def get_schema_field(self, field):
        i18n = II18n(field)
        result = self.field_factory(title=i18n.query_attribute('label'),
                                    description=i18n.query_attribute('description'),
                                    required=field.required,
                                    default=i18n.query_attribute('default'),
                                    value_type=Choice(
                                        vocabulary=ContactThemesListFieldValuesVocabulary(field)))
        result.__name__ = field.name
        return result


#
# Themes fields adapters
#

@factory_config(IContactThemeFieldInfo)
class ContactThemeFieldInfo(Persistent, Location):
    """Contact themes field information persistent class"""

    themes = FieldProperty(IContactThemeFieldInfo['themes'])
    display_mode = FieldProperty(IContactThemeFieldInfo['display_mode'])


@adapter_config(context=IContactThemeFieldInfo, provides=IFormContextPermissionChecker)
def contact_theme_field_permission_checker_factory(context):
    """Contact assignment permission checker"""
    form = get_parent(context, IWfForm)
    return IFormContextPermissionChecker(form)


@adapter_config(context=IFormField, provides=IContactThemeFieldInfo)
def contact_theme_form_field_info_factory(field):
    """Contact theme form field information factory"""
    if field.field_type not in ('contact_theme', 'contact_themes_list'):
        return None
    return get_annotation_adapter(field, CONTACT_THEME_FIELD_INFO_ANNOTATION_KEY,
                                  IContactThemeFieldInfo, name='++info++')


@vocabulary_config(name=CONTACT_THEME_FIELD_VALUES_VOCABULARY)
class ContactThemeFieldValuesVocabulary(SimpleVocabulary):
    """Contact themes field values vocabulary"""

    def __init__(self, context=None):
        terms = []
        field_info = IContactThemeFieldInfo(context, None)
        if field_info is not None:
            contacts = query_utility(IContactManager)
            if contacts is not None:
                manager = IContactThemeManager(contacts)
                request = query_request()
                for name in field_info.themes:
                    theme = manager.get(name)
                    if theme is not None:
                        terms.append(SimpleTerm(name, title=theme.get_label(request)))
                if context.required and len(terms) > 1:
                    terms.insert(0, SimpleTerm('',
                                               title=context.placeholder or
                                                     request.localizer.translate(
                                                         _("Please select a theme..."))))
        super().__init__(terms)


@vocabulary_config(name=CONTACT_THEMES_LIST_FIELD_VALUES_VOCABULARY)
class ContactThemesListFieldValuesVocabulary(SimpleVocabulary):
    """Contact themes list field values vocabulary"""

    def __init__(self, context=None):
        terms = []
        field_info = IContactThemeFieldInfo(context, None)
        if field_info is not None:
            contacts = query_utility(IContactManager)
            if contacts is not None:
                manager = IContactThemeManager(contacts)
                request = query_request()
                for name in field_info.themes:
                    theme = manager.get(name)
                    if theme is not None:
                        terms.append(SimpleTerm(name, title=theme.get_label(request)))
        super().__init__(terms)


@vocabulary_config(name=CONTACT_FORM_THEMES_FIELD_VALUES_VOCABULARY)
class ContactFormThemesFieldValuesVocabulary(SimpleVocabulary):
    """Contact form themes vocabulary"""

    def __init__(self, context=None):
        terms = []
        if IInternalFormLinkTheme.providedBy(context):
            context = context.__parent__
        if IInternalLink.providedBy(context):
            context = context.target
        if IWfForm.providedBy(context):
            has_themes, themes_field = boolean_iter(
                IFormFieldContainer(context).find_fields('contact_theme'))
            if has_themes:
                themes_field = next(themes_field)
                contacts = query_utility(IContactManager)
                if contacts is not None:
                    manager = IContactThemeManager(contacts)
                    request = query_request()
                    for name in IContactThemeFieldInfo(themes_field).themes:
                        theme = manager.get(name)
                        if theme is not None:
                            terms.append(SimpleTerm(name, title=theme.get_label(request)))
        super(ContactFormThemesFieldValuesVocabulary, self).__init__(terms)


@adapter_config(context=(IContactThemeField, IONFBaseLayer),
                provides=IFormFieldDataConverter)
class ContactThemeFieldDataConverter(ContextRequestAdapter):
    """Contact theme field data converter"""

    def convert(self, value):
        if value:
            contacts = query_utility(IContactManager)
            if contacts is not None:
                manager = IContactThemeManager(contacts)
                theme = manager.get(value)
                if theme is not None:
                    value = theme.get_label(self.request)
        return value


@adapter_config(context=(IContactThemesListField, IONFBaseLayer),
                provides=IFormFieldDataConverter)
class ContactThemesListFieldDataConverter(ContextRequestAdapter):
    """Contact themes list field data converter"""

    def convert(self, value):
        if value:
            if not isinstance(value, (list, tuple, set)):
                value = [value]
            else:
                value = list(value)
            contacts = query_utility(IContactManager)
            if contacts is not None:
                manager = IContactThemeManager(contacts)
                for index, val in enumerate(value):
                    theme = manager.get(val)
                    if theme is not None:
                        value[index] = theme.get_label(self.request)
        return value
