#
# Copyright (c) 2015-2020 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from collections import OrderedDict

from zope.contentprovider.interfaces import IContentProvider
from zope.interface import Attribute, Interface
from zope.schema import Bool, Choice
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from onf_website.shared.expengine import EXPENGINE_CONTENT_TYPE
from pyams_file.schema import ImageField
from pyams_i18n.schema import I18nTextField, I18nTextLineField
from pyams_sequence.schema import InternalReferenceField
from pyams_utils.schema import ColorField


__docformat__ = 'restructuredtext'

from onf_website import _


#
# Root header with experience engine
#

ORDERED_SELECTION_MODE = 'ordered'
RANDOM_SELECTION_MODE = 'random'

SELECTION_MODES = OrderedDict((
    (ORDERED_SELECTION_MODE, _("First gallery image")),
    (RANDOM_SELECTION_MODE, _("Random gallery image"))
))

SELECTION_MODES_VOCABULARY = SimpleVocabulary([
    SimpleTerm(k, title=v)
    for k, v in SELECTION_MODES.items()
])


THUMB_TEXT_ONLY_MODE = 'text'
THUMB_TARGET_ILLUSTRATION = 'illustration'
THUMB_TARGET_NAVIGATION = 'navigation'
THUMB_HIDDEN_MODE = 'hidden'

THUMBNAIL_MODES = OrderedDict((
    (THUMB_TEXT_ONLY_MODE, _("Text only, don't display thumbnail")),
    (THUMB_TARGET_ILLUSTRATION, _("Use target illustration")),
    (THUMB_TARGET_NAVIGATION, _("Use target navigation illustration")),
    (THUMB_HIDDEN_MODE, _("Hidden thumbnail"))
))

THUMBNAIL_MODES_VOCABULARY = SimpleVocabulary([
    SimpleTerm(k, title=v)
    for k, v in THUMBNAIL_MODES.items()
])


THUMB_BG_SAME_IMAGE = 'same'
THUMB_BG_TARGET_ILLUSTRATION = 'illustration'
THUMB_BG_TARGET_NAVIGATION = 'navigation'

THUMBNAIL_BG_MODES = OrderedDict((
    (THUMB_BG_SAME_IMAGE, _("Keep main illustration")),
    (THUMB_BG_TARGET_ILLUSTRATION, _("Use target illustration")),
    (THUMB_BG_TARGET_NAVIGATION, _("Use target navigation illustration"))
))

THUMBNAIL_BG_MODES_VOCABULARY = SimpleVocabulary([
    SimpleTerm(k, title=v)
    for k, v in THUMBNAIL_BG_MODES.items()
])


class IExperienceRootHeaderPortletRendererSettings(Interface):
    """Root header with illustration and experience engine"""

    images_selection_mode = Choice(title=_("Image selection"),
                                   description=_("This defines how main illustration will be "
                                                 "extracted from the medias gallery"),
                                   required=False,
                                   vocabulary=SELECTION_MODES_VOCABULARY,
                                   default=ORDERED_SELECTION_MODE)

    expengine = InternalReferenceField(title=_("Selected engine"),
                                       description=_("Reference to selected experiential engine"),
                                       content_type=EXPENGINE_CONTENT_TYPE,
                                       required=False)

    heading_text = I18nTextLineField(title=_("Heading text"),
                                     description=_("Text associated with this header; you can "
                                                   "use {brackets} to select text which will use "
                                                   "selected highlight color instead of default "
                                                   "color; you can also use pipes to force lines "
                                                   "breaks"),
                                     required=False)

    heading_base_color = ColorField(title=_("Heading base color"),
                                    description=_("This color will be used to display base "
                                                  "heading text"),
                                    required=False,
                                    default='ffffff')

    heading_highlight_color = ColorField(title=_("Heading highlight color"),
                                         description=_("This color will be used to display text "
                                                       "marked as highlighted using {brackets}"),
                                         required=False,
                                         default='fdf559')

    highlight_reference = InternalReferenceField(title=_("Highlight target"),
                                                 description=_("Internal reference to "
                                                               "highlighted content"),
                                                 required=False)

    highlight_target = Attribute("Highlight target content")

    thumbnail_display = Choice(title=_("Thumbnail display"),
                               description=_("You can choose which content is displayed into "
                                             "thumbnail content"),
                               required=True,
                               vocabulary=THUMBNAIL_MODES_VOCABULARY,
                               default=THUMB_TEXT_ONLY_MODE)

    background_selection_mode = Choice(title=_("Background selection"),
                                       description=_("You can choose which image is displayed in background when "
                                                     "mouse is placed above the thumbnail"),
                                       required=True,
                                       vocabulary=THUMBNAIL_BG_MODES_VOCABULARY,
                                       default=THUMB_BG_SAME_IMAGE)

    thumbnail_text = I18nTextLineField(title=_("Thumbnail text"),
                                       description=_("This text will be displayed into the "
                                                     "thumbnail if text mode is selected"),
                                       required=False)

    thumbnail_bg_color = ColorField(title=_("Thumbnail background color"),
                                    description=_("This is the thumbnail background color in "
                                                  "text mode"),
                                    required=False,
                                    default='002923')

    thumbnail_text_color = ColorField(title=_("Thumbnail text color"),
                                      description=_("This is the thumbnail text color in text "
                                                    "mode"),
                                      required=False,
                                      default='ffffff')

    thumbnail_alt_title = I18nTextLineField(title=_("Alternate title"),
                                            description=_("This alternate title will be used "
                                                          "instead of target title"),
                                            required=False)

    thumbnail_alt_header = I18nTextField(title=_("Alternate header"),
                                         description=_("This alternate header will be used "
                                                       "instead of target header"),
                                         required=False)

    thumbnail_mobile_header = I18nTextField(title=_("Mobile header"),
                                            description=_("This alternate header will be used "
                                                          "on mobile devices instead of standard "
                                                          "header"),
                                            required=False)


#
# Custom head portlet renderers interfaces
#

class IHubHeaderWithImagesPortletRendererSettings(Interface):
    """Hub header with images portlet renderer settings"""

    background_image = ImageField(title=_("Background image"),
                                  required=False)

    foreground_left_image = ImageField(title=_("Foreground left image"),
                                       required=False)

    foreground_right_image = ImageField(title=_("Foreground right image"),
                                        required=False)

    heading = I18nTextLineField(title=_('header-heading', default="Heading"),
                                description=_("Heading displayed on top above illustration"),
                                required=False)


#
# Content dard interface
#

class IContentCard(IContentProvider):
    """Content card interface"""
