#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import random

from persistent import Persistent
from pyramid.decorator import reify
from pyramid.traversal import lineage
from zope.container.contained import Contained
from zope.interface import Interface, implementer
from zope.schema.fieldproperty import FieldProperty

from onf_website.shared.common.skin.interfaces import \
    IExperienceRootHeaderPortletRendererSettings, RANDOM_SELECTION_MODE, THUMB_BG_SAME_IMAGE, \
    THUMB_BG_TARGET_ILLUSTRATION, \
    THUMB_BG_TARGET_NAVIGATION, \
    THUMB_TARGET_ILLUSTRATION, THUMB_TARGET_NAVIGATION, THUMB_TEXT_ONLY_MODE
from onf_website.skin.public import home_banner_anim, jquery_tilt, selector
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_content.component.gallery import IGallery
from pyams_content.component.illustration import IIllustration, ILinkIllustration
from pyams_content.shared.common.portlet.interfaces import ISharedContentHeaderPortletSettings
from pyams_default_theme.interfaces import IContentBannerIllustration, \
    IContentNavigationIllustration
from pyams_default_theme.shared.common.portlet.head import SharedContentHeaderPortletRenderer
from pyams_file.interfaces import IThumbnails
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_sequence.interfaces import IInternalReferencesList
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.text import break_text


__docformat__ = 'restructuredtext'

from onf_website import _


@adapter_config(context=(IPortalContext, IPyAMSLayer, Interface,
                         ISharedContentHeaderPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/head.pt', layer=IONFBaseLayer)
class BaseHeaderPortletRenderer(SharedContentHeaderPortletRenderer):
    """Base header portlet renderer"""

    label = _("ONF: Standard header with banner (default)")
    illustration_interface = IContentBannerIllustration

    @property
    def illustration(self):
        if not self.settings.display_illustration:
            return None, None, None
        registry = self.request.registry
        illustration = registry.queryMultiAdapter((self.context, self.request),
                                                  self.illustration_interface)
        if (illustration is None) or not illustration.has_data():
            return None, None, None
        data = II18n(illustration).query_attribute('data', request=self.request)
        if data.content_type.startswith('video'):
            return illustration, data, 'video'
        return illustration, data, 'illustration'


#
# Site root head portlet renderer
#

@adapter_config(name='ONF::head-root',
                context=(IPortalContext, IPyAMSLayer, Interface,
                         ISharedContentHeaderPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/head-root.pt', layer=IONFBaseLayer)
class RootHeaderPortletRenderer(BaseHeaderPortletRenderer):
    """Root header portlet renderer"""

    label = _("ONF: Site root header with panoramic illustrations")
    weight = 10


#
# Site root head portlet with included experience engine
#

@factory_config(IExperienceRootHeaderPortletRendererSettings)
@implementer(IInternalReferencesList)
class ExperienceRootHeaderPortletRendererSettings(Persistent, Contained):
    """Root header portlet renderer with experience engine settings"""

    images_selection_mode = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['images_selection_mode'])
    expengine = FieldProperty(IExperienceRootHeaderPortletRendererSettings['expengine'])
    heading_text = FieldProperty(IExperienceRootHeaderPortletRendererSettings['heading_text'])
    heading_base_color = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['heading_base_color'])
    heading_highlight_color = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['heading_highlight_color'])
    highlight_reference = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['highlight_reference'])
    thumbnail_display = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['thumbnail_display'])
    background_selection_mode = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['background_selection_mode'])
    thumbnail_text = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['thumbnail_text'])
    thumbnail_bg_color = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['thumbnail_bg_color'])
    thumbnail_text_color = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['thumbnail_text_color'])
    thumbnail_alt_title = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['thumbnail_alt_title'])
    thumbnail_alt_header = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['thumbnail_alt_header'])
    thumbnail_mobile_header = FieldProperty(
        IExperienceRootHeaderPortletRendererSettings['thumbnail_mobile_header'])

    @property
    def engine_target(self):
        """Experiential engine target"""
        return get_reference_target(self.expengine)

    @property
    def highlight_target(self):
        """Highlighted content target"""
        return get_reference_target(self.highlight_reference)

    @property
    def references(self):
        return filter(bool, (self.expengine, self.highlight_reference))


@adapter_config(name='ONF::head-root-exp',
                context=(IPortalContext, IPyAMSLayer, Interface,
                         ISharedContentHeaderPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/head-root-exp.pt', layer=IONFBaseLayer)
class ExperienceRootHeaderPortletRenderer(BaseHeaderPortletRenderer):
    """Root header portlet renderer with experience engine"""

    label = _("ONF: Site root header with illustration and experience engine")
    weight = 11

    settings_interface = IExperienceRootHeaderPortletRendererSettings
    resources = (selector, home_banner_anim)

    @reify
    def highlight_target(self):
        return self.renderer_settings.highlight_target

    def get_images(self):
        """Images gallery getter"""
        images = None
        for level in lineage(self.context):
            gallery = IGallery(level, None)
            if gallery is None:
                continue
            images = list(gallery.get_visible_images())
            if not images:
                continue
            break
        if not images:
            return
        settings = self.renderer_settings
        if settings.images_selection_mode == RANDOM_SELECTION_MODE:
            random.shuffle(images)
        first_image = images[0]
        last_image = first_image
        # check illustration
        if settings.background_selection_mode != THUMB_BG_SAME_IMAGE:
            target = self.highlight_target
            if target is not None:
                illustration = None
                if settings.background_selection_mode == THUMB_BG_TARGET_ILLUSTRATION:
                    illustration = IIllustration(target, None)
                elif settings.background_selection_mode == THUMB_BG_TARGET_NAVIGATION:
                    request = self.request
                    registry = request.registry
                    illustration = registry.queryMultiAdapter((target, request),
                                                              IContentNavigationIllustration)
                if (illustration is not None) and illustration.has_data():
                    last_image = illustration
        yield from zip(('media', 'submedia'), (first_image, last_image))

    def get_heading(self):
        """Heading text getter"""
        settings = self.renderer_settings
        heading = II18n(settings).query_attribute('heading_text', request=self.request)
        if not heading:
            return ''
        return '<span style="color: #{};">{}</span>'.format(
            settings.heading_base_color,
            break_text(heading)
                    .replace('{',
                             '<strong style="color: #{};">'.format(
                                 settings.heading_highlight_color))
                    .replace('}',
                             '</strong>'))

    def get_highlight_image(self, default):
        """Get highlight image"""
        image = default
        settings = self.renderer_settings
        if settings.thumbnail_display != THUMB_TEXT_ONLY_MODE:
            request = self.request
            registry = request.registry
            target = self.highlight_target
            illustration = None
            if settings.thumbnail_display == THUMB_TARGET_ILLUSTRATION:
                illustration = IIllustration(target, None)
            elif settings.thumbnail_display == THUMB_TARGET_NAVIGATION:
                illustration = registry.queryMultiAdapter((target, request),
                                                          IContentNavigationIllustration)
            if illustration is not None:
                image = II18n(illustration).query_attribute('data', request=request)
        if image is not None:
            thumbnails = IThumbnails(image, None)
            if thumbnails is not None:
                return thumbnails.get_thumbnail('square')
        return None


#
# Forest header portlet renderer
#

@adapter_config(name='ONF::head-forest',
                context=(IPortalContext, IPyAMSLayer, Interface,
                         ISharedContentHeaderPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/head-forest.pt', layer=IONFBaseLayer)
class ForestHeaderPortletRenderer(BaseHeaderPortletRenderer):
    """Forest header portlet renderer"""

    label = _("ONF: Forest header with illustration")
    weight = 50

    resources = (jquery_tilt, )
