#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectRemovedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing.interfaces import IHearingManagerTarget
from onf_website.component.location import ILocationManagerTarget
from onf_website.shared.casestudy import CaseStudy
from onf_website.shared.casestudy.interfaces import CASE_STUDY_CONTENT_TYPE, ICaseStudyManager, ICaseStudyManagerFactory
from onf_website.shared.common.security import SharedToolManagerRestrictionInfo
from pyams_content.component.paragraph.interfaces import IParagraphFactorySettings
from pyams_content.component.theme.interfaces import IThemesManagerTarget
from pyams_content.reference.pictograms.interfaces import IPictogramManagerTarget
from pyams_content.shared.common.interfaces import IManagerRestrictionsFactory, ISharedContentFactory
from pyams_content.shared.common.manager import SharedTool
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import utility_config
from pyams_utils.traversing import get_parent


@implementer(ICaseStudyManager, IParagraphFactorySettings, ILocationManagerTarget, IHearingManagerTarget,
             IThemesManagerTarget, IPictogramManagerTarget)
class CaseStudyManager(SharedTool):
    """Case study manager class"""

    shared_content_type = CASE_STUDY_CONTENT_TYPE

    allowed_paragraphs = FieldProperty(IParagraphFactorySettings['allowed_paragraphs'])
    auto_created_paragraphs = FieldProperty(IParagraphFactorySettings['auto_created_paragraphs'])


@utility_config(provides=ICaseStudyManagerFactory)
class CaseStudyManagerFactory(object):
    """Default case study manager factory"""

    def __new__(cls):
        return CaseStudyManager


@adapter_config(context=ICaseStudyManager, provides=IManagerRestrictionsFactory)
def case_study_manager_restrictions_factory(context):
    """Case study manager custom restrictions factory"""
    return SharedToolManagerRestrictionInfo


@adapter_config(context=ICaseStudyManager, provides=ISharedContentFactory)
def case_study_manager_content_factory(context):
    return CaseStudy


@subscriber(IObjectAddedEvent, context_selector=ICaseStudyManager)
def handle_added_case_study_manager(event):
    """Register case study manager when added"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, ICaseStudyManager)


@subscriber(IObjectRemovedEvent, context_selector=ICaseStudyManager)
def handle_removed_case_study_manager(event):
    """Unregister case study manager when removed"""
    site = get_parent(event.oldParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.unregisterUtility(event.object, ICaseStudyManager)
