#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from colander import Boolean, DateTime, Int, MappingSchema, OneOf, SchemaNode, SequenceSchema, String, drop

from onf_website.component.location.api.schema import LocationSetter
from pyams_content.shared.alert.api.schema import AlertInfo
from pyams_content.workflow.basic import ARCHIVED, DRAFT, PUBLISHED, STATES_IDS
from pyams_utils.rest import BaseResponseSchema


class AlertsQuerySchema(MappingSchema):
    """Alerts query schema"""
    id_nat_frt = SchemaNode(String(),
                            description="Forest national ID location of alerts; you can set several forests "
                                        "by separating forests IDs with commas",
                            missing=drop)


class InternalAlertsQuerySchema(MappingSchema):
    """Internal alerts query schema"""
    id_nat_frt = SchemaNode(String(),
                            description="Forest national ID location of alerts; you can set several forests "
                                        "by separating forests IDs with commas",
                            missing=drop)
    state = SchemaNode(String(),
                       description="Workflow publication state; you can query multiple states using a "
                                   "comma separator",
                       missing=drop)
    owner = SchemaNode(String(),
                       description="Alerts owner; if not set, only current owner alerts are returned",
                       missing=drop)
    age_limit = SchemaNode(Int(),
                           description="Maximum days from content publication date, in days",
                           missing=drop)


class NewAlertSchema(MappingSchema):
    """New alert schema"""
    alert_type = SchemaNode(String(),
                            description="Alert type")
    title = SchemaNode(String(),
                       description="Alert title")
    body = SchemaNode(String(),
                      description="Alert message body",
                      missing=drop)
    location = LocationSetter(description="Alert location",
                              missing=drop)
    href = SchemaNode(String(),
                      description="Optional link URL",
                      missing=drop)
    state = SchemaNode(String(),
                       description="State of created alert; if state is set to 'published' without a publication "
                                   "date, publication is effective immediately",
                       validator=OneOf((DRAFT, PUBLISHED, ARCHIVED)),
                       missing=DRAFT)
    notify = SchemaNode(Boolean(),
                        description="If false, notifications will not be sent to users which subscribed to alerts "
                                    "matching new alert location",
                        missing=True)
    publication_date = SchemaNode(DateTime(),
                                  description="First publication datetime in iso8601 format, including timezone; "
                                              "if state is set to published and publication date is not set, "
                                              "the alert will be published immediately",
                                  missing=drop)
    expiration_date = SchemaNode(DateTime(),
                                 description="Expiration date in iso8601 format, including timezone; setting "
                                             "expiration date is useless without publication date, and is only "
                                             "applied if state is changed",
                                 missing=drop)
    maximum_interval = SchemaNode(Int(),
                                  description="Maximum display interval, in hours",
                                  missing=48)


class AlertUpdateSchema(NewAlertSchema):
    """Alert update schema"""
    alert_type = SchemaNode(String(),
                            description="Alert type",
                            missing=drop)
    title = SchemaNode(String(),
                       description="Alert title",
                       missing=drop)
    state = SchemaNode(String(),
                       description="State alert; if state is set to 'published' without a publication date, "
                                   "publication is effective immediately",
                       validator=OneOf((DRAFT, PUBLISHED, ARCHIVED)),
                       missing=drop)
    notify = SchemaNode(Boolean(),
                        description="Notifications will be sent to users which subscribed to alerts matching "
                                    "alert location only if this argument is true",
                        missing=False)
    maximum_interval = SchemaNode(Int(),
                                  description="Maximum display interval, in hours",
                                  missing=drop)


class InternalAlertsResultsList(SequenceSchema):
    """Internal alerts results list"""
    content = AlertInfo()


class InternalAlertsQueryResponse(BaseResponseSchema):
    """Internal alerts query response"""
    results = InternalAlertsResultsList(description="Internal alerts search results",
                                        missing=drop)


class InternalAlertsGetterRequest(MappingSchema):
    """Internal alerts getter request"""
    querystring = InternalAlertsQuerySchema()


class InternalAlertsGetterResponse(MappingSchema):
    """Internal alerts getter response"""
    body = InternalAlertsQueryResponse()


class InternalAlertGetterQuery(MappingSchema):
    """Internal alert getter query"""
    state = SchemaNode(String(),
                       description="Workflow publication state",
                       validator=OneOf(STATES_IDS),
                       missing=drop)
    version = SchemaNode(Int(),
                         description="Workflow version number",
                         missing=drop)


class InternalAlertGetterRequest(MappingSchema):
    """Internal alert getter request"""
    querystring = InternalAlertGetterQuery()


class InternalAlertGetterInfo(BaseResponseSchema):
    """Alert getter response"""
    content = AlertInfo(description="Alert properties")


class InternalAlertGetterResponse(MappingSchema):
    """Alert getter response"""
    body = InternalAlertGetterInfo()


class AlertCreationRequest(MappingSchema):
    """Alert creation request"""
    body = NewAlertSchema(description="New alert properties")


class AlertCreationStatus(BaseResponseSchema):
    """Alert creation status"""
    content = AlertInfo(description="New alert properties")


class AlertCreationResponse(MappingSchema):
    """Alert creation response"""
    body = AlertCreationStatus()


class AlertUpdateRequest(MappingSchema):
    """Alert update request"""
    body = AlertUpdateSchema(description="Alert update properties")


class AlertUpdateStatus(BaseResponseSchema):
    """Alert update status"""
    content = AlertInfo(description="Updated alert properties")


class AlertUpdateResponse(MappingSchema):
    """Alert update response"""
    body = AlertUpdateStatus()


class AlertDeleteStatus(BaseResponseSchema):
    """Alert delete status"""


class AlertDeleteResponse(MappingSchema):
    """Alert delete response"""
    body = AlertDeleteStatus()
