#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from cornice import Service
from cornice.validators import colander_validator
from pyramid.httpexceptions import HTTPBadRequest, HTTPCreated, HTTPForbidden, HTTPNotFound, HTTPNotModified, HTTPOk, \
    HTTPServiceUnavailable, HTTPUnauthorized
from pyramid.security import Authenticated
from zope.schema.interfaces import ValidationError

from onf_website.shared.alert.api.interfaces import REST_ALERTS_INTERNAL_ROUTE, REST_ALERT_INTERNAL_ROUTE
from onf_website.shared.alert.api.schema import AlertCreationRequest, AlertCreationResponse, AlertDeleteResponse, \
    AlertUpdateRequest, AlertUpdateResponse, InternalAlertGetterRequest, InternalAlertGetterResponse, \
    InternalAlertsGetterRequest, InternalAlertsGetterResponse
from pyams_content.interfaces import CREATE_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.alert import IAlert
from pyams_content.shared.alert.interfaces import IAlertsManager
from pyams_content.shared.common.interfaces import WfSharedContentNotificationEvent
from pyams_content.workflow.basic import ARCHIVED, DRAFT, PUBLISHED, STATES_IDS
from pyams_security.rest import check_cors_origin, set_cors_headers
from pyams_sequence.reference import get_last_version, get_reference_target
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import query_utility
from pyams_utils.rest import http_error, http_response, rest_responses
from pyams_utils.traversing import get_parent
from pyams_workflow.interfaces import AmbiguousTransitionError, IWorkflow, IWorkflowInfo, IWorkflowState, \
    IWorkflowVersions, InvalidTransitionError, VersionError


__docformat__ = 'restructuredtext'


#
# Internal alerts query service
#

internal_alerts_service = Service(name=REST_ALERTS_INTERNAL_ROUTE,
                                  pyramid_route=REST_ALERTS_INTERNAL_ROUTE,
                                  description="ONF website internal alerts service")


@internal_alerts_service.options(validators=(check_cors_origin, set_cors_headers))
def internal_alerts_service_options(request):
    return ''


internal_alerts_getter_responses = rest_responses.copy()
internal_alerts_getter_responses[HTTPOk.code] = InternalAlertsGetterResponse(
    description="Internal alerts search results")


@internal_alerts_service.get(schema=InternalAlertsGetterRequest(),
                             validators=(check_cors_origin, colander_validator, set_cors_headers),
                             response_schemas=internal_alerts_getter_responses)
def get_internal_alerts(request):
    """Internal alerts search service"""
    alerts = query_utility(IAlertsManager)
    if alerts is None:
        return http_error(request, HTTPServiceUnavailable)
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    if not request.has_permission(VIEW_SYSTEM_PERMISSION, context=alerts):
        return http_error(request, HTTPForbidden)
    params = request.validated.get('querystring', {})
    if 'state' in params:
        state = set(filter(bool, params['state'].split(',')))
        if state - set(STATES_IDS):
            return http_error(request, HTTPBadRequest, "Invalid state.")
    return http_response(request, HTTPOk, {
        'results': [
            alert.to_json(params)
            for alert in alerts.find(params, request)
        ]
    })


#
# Alert creation service
#

alert_creation_responses = rest_responses.copy()
alert_creation_responses[HTTPCreated.code] = AlertCreationResponse(description="Alert creation response")


@internal_alerts_service.post(schema=AlertCreationRequest(),
                              validators=(check_cors_origin, colander_validator, set_cors_headers),
                              require_csrf=False,
                              response_schemas=alert_creation_responses)
def create_alert(request):
    """Alert creation service"""
    alerts = query_utility(IAlertsManager)
    if alerts is None:
        return http_error(request, HTTPServiceUnavailable)
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    if not request.has_permission(CREATE_CONTENT_PERMISSION, context=alerts):
        return http_error(request, HTTPForbidden)
    params = request.validated.get('body', {})
    try:
        alert = alerts.create(params, request)
    except ValidationError as ex:
        return http_error(request, HTTPBadRequest, "Validation error: {}!".format(ex))
    except AttributeError as ex:
        return http_error(request, HTTPBadRequest, "Attribute error: {}!".format(ex))
    except ValueError as ex:
        return http_error(request, HTTPBadRequest, "Value error: {}!".format(ex))
    except AmbiguousTransitionError as ex:
        return http_error(request, HTTPBadRequest, "Ambiguous transition: {}!".format(ex))
    except InvalidTransitionError as ex:
        return http_error(request, HTTPBadRequest, "Invalid transition: {}!".format(ex))
    except:
        return http_error(request, HTTPBadRequest)
    else:
        if params.get('notify', False):
            workflow = IWorkflow(alert)
            state = IWorkflowState(alert)
            if state.state in workflow.published_states:
                request.registry.notify(WfSharedContentNotificationEvent(alert, request))
        return http_response(request, HTTPCreated, {
            'content': alert.to_json(params)
        })


#
# Alert update service
#

alert_update_service = Service(name=REST_ALERT_INTERNAL_ROUTE,
                               pyramid_route=REST_ALERT_INTERNAL_ROUTE,
                               description='ONF website alerts API')


alert_update_getter_responses = rest_responses.copy()
alert_update_getter_responses[HTTPOk.code] = InternalAlertGetterResponse(description="Alert getter response")


@alert_update_service.get(schema=InternalAlertGetterRequest(),
                          validators=(check_cors_origin, colander_validator, set_cors_headers),
                          require_csrf=False,
                          response_schemas=alert_update_getter_responses)
def get_alert(request):
    """Alert getter service"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    oid = request.matchdict['oid']
    if not oid:
        return http_error(request, HTTPBadRequest)
    alert = None
    params = request.validated.get('querystring', {})
    version = params.get('version')
    state = params.get('state')
    if version:
        target = get_reference_target(oid, request=request, getter=get_last_version)
        versions = IWorkflowVersions(target, None)
        if versions is not None:
            try:
                alert = versions.get_version(version)
            except VersionError:
                pass
    elif state:
        alert = get_reference_target(oid, request=request, state=state)
    else:
        alert = get_reference_target(oid, request=request, getter=get_last_version)
    if alert is None:
        return http_error(request, HTTPNotFound)
    if not request.has_permission(VIEW_SYSTEM_PERMISSION, context=alert):
        return http_error(request, HTTPForbidden)
    return http_response(request, HTTPOk, {
        'content': alert.to_json(params)
    })


alert_update_patch_responses = rest_responses.copy()
alert_update_patch_responses[HTTPOk.code] = AlertUpdateResponse(description="Alert update response")


@alert_update_service.patch(schema=AlertUpdateRequest(),
                            validators=(check_cors_origin, colander_validator, set_cors_headers),
                            require_csrf=False,
                            response_schemas=alert_update_patch_responses)
def update_alert(request):
    """Alert update service"""
    alerts = query_utility(IAlertsManager)
    if alerts is None:
        return http_error(request, HTTPServiceUnavailable)
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    oid = request.matchdict['oid']
    if not oid:
        return http_error(request, HTTPBadRequest)
    alert = get_reference_target(oid, request=request, getter=get_last_version)
    if alert is None:
        return http_error(request, HTTPNotFound)
    if not request.has_permission(MANAGE_CONTENT_PERMISSION, context=alert):
        return http_error(request, HTTPForbidden)
    params = request.validated.get('body', {})
    try:
        version = alert.update_from_json(params, request)
    except ValidationError as ex:
        return http_error(request, HTTPBadRequest, "Validation error: {}!".format(ex))
    except AttributeError as ex:
        return http_error(request, HTTPBadRequest, "Attribute error: {}!".format(ex))
    except ValueError as ex:
        return http_error(request, HTTPBadRequest, "Value error: {}!".format(ex))
    except AmbiguousTransitionError as ex:
        return http_error(request, HTTPBadRequest, "Ambiguous transition: {}!".format(ex))
    except InvalidTransitionError as ex:
        return http_error(request, HTTPBadRequest, "Invalid transition: {}!".format(ex))
    except:
        return http_error(request, HTTPBadRequest)
    else:
        if params.get('notify', False):
            workflow = IWorkflow(alert)
            state = IWorkflowState(alert)
            if state.state in workflow.published_states:
                request.registry.notify(WfSharedContentNotificationEvent(version, request))
        return http_response(request, HTTPOk, {
            'content': version.to_json(params)
        })


alert_delete_responses = rest_responses.copy()
alert_delete_responses[HTTPOk.code] = AlertDeleteResponse(description="Alert delete response")
alert_delete_responses[HTTPNotModified.code] = AlertDeleteResponse(description="Alert was not modified")


@alert_update_service.delete(validators=(check_cors_origin, set_cors_headers),
                             require_csrf=False,
                             response_schemas=alert_delete_responses)
def delete_alert(request):
    """Alert delete service"""
    alerts = query_utility(IAlertsManager)
    if alerts is None:
        return http_error(request, HTTPServiceUnavailable)
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    oid = request.matchdict['oid']
    if not oid:
        return http_error(request, HTTPBadRequest)
    alert = get_reference_target(oid, request=request, getter=get_last_version)
    if alert is None:
        return http_error(request, HTTPNotFound)
    if not request.has_permission(MANAGE_CONTENT_PERMISSION, context=alert):
        return http_error(request, HTTPForbidden)
    wf_state = IWorkflowState(alert)
    if wf_state.state == DRAFT:
        if wf_state.version_id == 1:  # delete draft version
            parent = get_parent(alert, IAlert)
            del alerts[parent.__name__]
            return http_response(request, HTTPOk, {
                'message': "Alert deleted"
            })
        versions = IWorkflowVersions(alert)
        versions.remove_version(wf_state.version_id)
        return http_response(request, HTTPOk, {
            'message': "Alert version {} deleted".format(wf_state.version_id)
        })
    elif wf_state.state == PUBLISHED:
        wf_info = IWorkflowInfo(alert)
        wf_info.fire_transition_toward(ARCHIVED, check_security=False, principal=request.principal.id)
        return http_response(request, HTTPOk, {
            'message': "Alert version {} archived".format(wf_state.version_id)
        })
    return http_response(request, HTTPNotModified, {
        'message': "Alert was not modified"
    })
