#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from persistent import Persistent
from pyramid.events import subscriber
from zope.container.contained import Contained
from zope.interface import implementer, provider
from zope.lifecycleevent.interfaces import IObjectCreatedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing import IHearingInfo, IHearingTarget
from onf_website.component.location import ILocationTarget
from onf_website.shared.activity.interfaces import ACTIVITY_CONTENT_NAME, ACTIVITY_CONTENT_TYPE, \
    ACTIVITY_INFO_ANNOTATIONS_KEY, IActivity, IActivityInfo, IActivityManager, IWfActivity, \
    IWfActivityFactory
from pyams_content.component.gallery import IGalleryTarget
from pyams_content.component.illustration import IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.component.paragraph import IParagraphContainerTarget
from pyams_content.component.theme import ITagsTarget, IThemesTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.review import IReviewTarget
from pyams_content.shared.common import IWfSharedContentFactory, SharedContent, WfSharedContent, \
    register_content_type
from pyams_content.shared.common.types import WfTypedSharedContentMixin
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config


__docformat__ = 'restructuredtext'

from pyams_utils.registry import get_utility


@factory_config(IWfActivity)
@implementer(IWfActivity, IIllustrationTarget, ILinkIllustrationTarget,
             IGalleryTarget, IParagraphContainerTarget,
             ILocationTarget, IHearingTarget, ITagsTarget, IThemesTarget,
             IPreviewTarget, IReviewTarget)
class WfActivity(WfSharedContent, WfTypedSharedContentMixin):
    """Activity class"""

    content_type = ACTIVITY_CONTENT_TYPE
    content_name = ACTIVITY_CONTENT_NAME

    references = FieldProperty(IWfActivity['references'])


register_content_type(WfActivity)


@factory_config(IActivityInfo)
class ActivityInfo(Persistent, Contained):
    """Activity persistent information"""

    markup = FieldProperty(IActivityInfo['markup'])
    duration = FieldProperty(IActivityInfo['duration'])
    length = FieldProperty(IActivityInfo['length'])
    altitude = FieldProperty(IActivityInfo['altitude'])
    price = FieldProperty(IActivityInfo['price'])


@adapter_config(context=IWfActivity, provides=IActivityInfo)
def activity_info_factory(context):
    """Activity info factory"""
    return get_annotation_adapter(context, ACTIVITY_INFO_ANNOTATIONS_KEY, IActivityInfo,
                                  name='++info++')


@provider(IWfActivityFactory)
@implementer(IActivity)
class Activity(SharedContent):
    """Workflow managed activity"""


@adapter_config(context=IWfActivityFactory, provides=IWfSharedContentFactory)
def activity_content_factory(context):
    return WfActivity


@subscriber(IObjectCreatedEvent, context_selector=IWfActivity)
def handle_new_activity(event):
    """Handle new activity"""
    manager = get_utility(IActivityManager)
    activity = event.object
    hearing = IHearingInfo(activity)
    hearing.source = manager.default_hearing_source
    hearing.source_folder = manager.default_source_folder
