#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json
import locale

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import field
from z3c.table.interfaces import IColumn
from zope.interface import implementer

from onf_website.reference.universe import Universe
from onf_website.reference.universe.interfaces import IUniverse, IUniverseTable
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference.zmi import ReferenceInfoAJAXEditForm
from pyams_content.reference.zmi.table import ReferenceTableContentsTable, ReferenceTableContentsView, \
    ReferenceTablePropertiesEditForm
from pyams_content.reference.zmi.table import ReferenceTablePropertiesMenu
from pyams_content.shared.common.interfaces import ISharedSite
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IInnerPage
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import SorterColumn
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import get_all_utilities_registered_for
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm, AdminEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer

__docformat__ = 'restructuredtext'

from onf_website import _


#
# Universe table views
#

class UniverseTableContentsTable(ReferenceTableContentsTable):
    """Universe table contents table"""

    hide_body_toolbar = True
    sortOn = None
    permission = MANAGE_SITE_ROOT_PERMISSION

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super(ReferenceTableContentsTable, self).data_attributes
        attributes['table'] = {
            'id': self.id,
            'data-ams-location': absolute_url(self.context, self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-univers': 'set-universe-order.json'
        }
        return attributes


@adapter_config(name='sorter',
                context=(IUniverseTable, IPyAMSLayer, UniverseTableContentsTable),
                provides=IColumn)
class UniverseTableSorterColumn(SorterColumn):
    """Universe table sorter column"""


@pagelet_config(name='contents.html',
                context=IUniverseTable,
                layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class UniverseTableContentsView(ReferenceTableContentsView):
    """Universe table contents view"""

    table_class = UniverseTableContentsTable


@view_config(name='set-universe-order.json',
             context=IUniverseTable, request_type=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION,
             renderer='json', xhr=True)
def set_universe_order(request):
    """Update universe order"""
    container = IUniverseTable(request.context)
    order = list(map(str, json.loads(request.params.get('names'))))
    container.updateOrder(order)
    return {'status': 'success'}


#
# Universs views
#

@viewlet_config(name='add-universe.action',
                context=IUniverseTable, layer=IAdminLayer, view=UniverseTableContentsTable,
                manager=IWidgetTitleViewletManager,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class UniverseAddAction(ToolbarAction):
    """Universe add action"""

    label = _("Add universe")

    url = 'add-universe.html'
    modal_target = True


@pagelet_config(name='add-universe.html',
                context=IUniverseTable, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='add-univers.json',
             context=IUniverseTable, layer=IPyAMSLayer,
             base=AJAXAddForm)
class UniverseAddForm(AdminDialogAddForm):
    """Universe add form"""

    legend = _("Adding new universe")

    fields = field.Fields(IUniverse).omit('__parent__')
    edit_permission = MANAGE_SITE_ROOT_PERMISSION
    
    def update(self):
        super().update()
        context = self.context
        request = self.request
        parent = self.__parent__
        for site in sorted(get_all_utilities_registered_for(ISharedSite),
                           key=lambda x: locale.strxfrm(II18n(x).query_attribute('title',
                                                                                 request=request)
                                                        or '')):
            print("site")
        
    def create(self, data):
        return Universe()

    def update_content(self, content, data):
        form_data = data.get(self, data)
        form_data['__name__'] = translate_string(form_data['__name__'], spaces='-', keep_chars='-')
        return super().update_content(content, data)

    def add(self, object):
        self.context[object.__name__] = object

    def nextURL(self):
        return 'contents.html'


@pagelet_config(name='properties.html',
                context=IUniverse, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class UniversEditForm(AdminDialogEditForm):
    """Universe edit form"""

    prefix = 'target_properties.'

    legend = _("Edit univers properties")

    fields = field.Fields(IUniverse).omit('__parent__', '__name__')

    ajax_handler = 'properties.json'
    edit_permission = MANAGE_SITE_ROOT_PERMISSION


@view_config(name='properties.json',
             context=IUniverse, request_type=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION,
             renderer='json', xhr=True)
class UniversAJAXEditForm(ReferenceInfoAJAXEditForm, UniversEditForm):
    """Universe edit form, JSON renderer"""

    table_factory = UniverseTableContentsTable


#
# Universe table properties
#

@pagelet_config(name='properties.html',
                context=IUniverseTable, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IUniverseTable, layer=IPyAMSLayer)
class UniversTablePropertiesEditForm(ReferenceTablePropertiesEditForm):
    """Universe table properties edit form"""

    fields = field.Fields(IUniverseTable).omit('__parent__', '__name__')
