#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from onf_website.reference.universe.interfaces import IUniverse, IUniverseTable
from pyams_content.reference import ReferenceInfo
from pyams_i18n.interfaces import II18n, II18nManager
from pyams_utils.container import BTreeOrderedContainer
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


@implementer(IUniverseTable, II18nManager)
class UniverseTable(BTreeOrderedContainer):
    """Universe table"""

    title = FieldProperty(IUniverseTable['title'])
    short_name = FieldProperty(IUniverseTable['short_name'])
    dropdown_menu_title = FieldProperty(IUniverseTable['dropdown_menu_title'])
    inactive_info_title = FieldProperty(IUniverseTable['inactive_info_title'])
    inactive_info_text = FieldProperty(IUniverseTable['inactive_info_text'])
    online_help_title = FieldProperty(IUniverseTable['online_help_title'])
    online_help_text = FieldProperty(IUniverseTable['online_help_text'])

    languages = FieldProperty(II18nManager['languages'])


@subscriber(IObjectAddedEvent, context_selector=IUniverseTable)
def handle_added_universe_table(event):
    """Handle new universe table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IUniverseTable)


@implementer(IUniverse)
class Universe(ReferenceInfo):
    """Universe record"""
    title = FieldProperty(IUniverse["title"])
    description = FieldProperty(IUniverse["description"])
    contact_target = FieldProperty(IUniverse["contact_target"])
    

@vocabulary_config(name='ONF universe')
class UniverseVocabulary(SimpleVocabulary):
    """Universs vocabulary"""

    def __init__(self, context=None):
        table = query_utility(IUniverseTable)
        if table is not None:
            request = check_request()
            terms = [SimpleTerm(v,
                                token=v.__name__,
                                title=II18n(v).query_attribute('title', request=request))
                     for v in table.values()]
        else:
            terms = []
        super().__init__(terms)
