#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import field
from z3c.table.interfaces import IColumn

from onf_website.reference.target import Target
from onf_website.reference.target.interfaces import ITarget, ITargetTable
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference.zmi import ReferenceInfoAJAXEditForm
from pyams_content.reference.zmi.table import ReferenceTableContentsTable, ReferenceTableContentsView
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import SorterColumn
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.layer import IAdminLayer

__docformat__ = 'restructuredtext'

from onf_website import _


#
# Targets table views
#

class TargetTableContentsTable(ReferenceTableContentsTable):
    """Target table contents table"""

    hide_body_toolbar = True
    sortOn = None
    permission = MANAGE_SITE_ROOT_PERMISSION

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super(ReferenceTableContentsTable, self).data_attributes
        attributes['table'] = {
            'id': self.id,
            'data-ams-location': absolute_url(self.context, self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-targets-order.json'
        }
        return attributes


@adapter_config(name='sorter', context=(ITargetTable, IPyAMSLayer, TargetTableContentsTable), provides=IColumn)
class TargetTableSorterColumn(SorterColumn):
    """Target table sorter column"""


@pagelet_config(name='contents.html', context=ITargetTable, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
class TargetTableContentsView(ReferenceTableContentsView):
    """Target table contents view"""

    table_class = TargetTableContentsTable


@view_config(name='set-targets-order.json', context=ITargetTable, request_type=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION, renderer='json', xhr=True)
def set_targets_order(request):
    """Update targets order"""
    container = ITargetTable(request.context)
    order = list(map(str, json.loads(request.params.get('names'))))
    container.updateOrder(order)
    return {'status': 'success'}


#
# Targets views
#

@viewlet_config(name='add-target.action', context=ITargetTable, layer=IAdminLayer,
                manager=IWidgetTitleViewletManager, view=TargetTableContentsTable,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class TargetAddAction(ToolbarAction):
    """Target add action"""

    label = _("Add target")

    url = 'add-target.html'
    modal_target = True


@pagelet_config(name='add-target.html', context=ITargetTable, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='add-target.json', context=ITargetTable, layer=IPyAMSLayer, base=AJAXAddForm)
class TargetAddForm(AdminDialogAddForm):
    """Target add form"""

    legend = _("Adding new target")

    fields = field.Fields(ITarget).omit('__parent__')
    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def create(self, data):
        return Target()

    def update_content(self, content, data):
        form_data = data.get(self, data)
        form_data['__name__'] = translate_string(form_data['__name__'], spaces='-', keep_chars='-')
        return super(TargetAddForm, self).update_content(content, data)

    def add(self, object):
        self.context[object.__name__] = object

    def nextURL(self):
        return 'contents.html'


@pagelet_config(name='properties.html', context=ITarget, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
class TargetEditForm(AdminDialogEditForm):
    """Target edit form"""

    prefix = 'target_properties.'

    legend = _("Edit target properties")

    fields = field.Fields(ITarget).omit('__parent__', '__name__')

    ajax_handler = 'properties.json'
    edit_permission = MANAGE_SITE_ROOT_PERMISSION


@view_config(name='properties.json', context=ITarget, request_type=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION, renderer='json', xhr=True)
class TargetAJAXEditForm(ReferenceInfoAJAXEditForm, TargetEditForm):
    """Target edit form, JSON renderer"""

    table_factory = TargetTableContentsTable
