
import re
from datetime import date

from sqlalchemy import Boolean, Column, Date, Integer, Numeric, Text, Unicode
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.orm import relation
from sqlalchemy.schema import ForeignKey, PrimaryKeyConstraint
from sqlalchemy.sql import and_
from zope.interface import implementer, provider
from zope.schema.interfaces import ITitledTokenizedTerm

from onf_website.reference.forest.model.foret import Foret, InformationForet
from onf_website.reference.planning.model import PARENT_SCHEMA, PARENT_SESSION
from onf_website.reference.planning.model.interfaces.planning import IInformationPlanning, \
    IModificationPlanning, IPlanningData, IPlanningModel
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin
from pyams_utils.unicode import translate_string
from onf_website.reference.insee.model import Departement
from sqlalchemy.orm import relationship
from sqlalchemy.schema import ForeignKey, ForeignKeyConstraint


# Polymorphic base classes

REG_ID_NAT_AMGT = re.compile('A([0-9]{6})[A-Z]')


@provider(IPlanningData)
@implementer(IPlanningModel, ITitledTokenizedTerm)
class PlanningData(DynamicSchemaMixin, Base):
    """Base model class for plannings"""

    __tablename__ = 'v_amenagement'
    __schema__ = PARENT_SCHEMA

    id_amenagement = Column(Integer, primary_key=True)
    amenagement_domanial = Column(Unicode(3))
    id_nat_amgt = Column('identifiant_nat_amenagement', Unicode(8))
    numero_modification	= Column(Integer)
    statut = Column(Unicode(20))
    categorie_proprietaire = Column(Unicode(80))
    code_categorie_proprietaire = Column(Unicode(2))
    nom_amenagement = Column(Unicode(128))
    nom_usage_amenagement = Column(Unicode(160))
    departement_situation = Column(Unicode(3))
    departement_situation_suppl = Column(Unicode(3))
    code_service = Column(Unicode(6))
    region_ifn = Column(Unicode(3))
    dra_sra = Column(Unicode(120))
    type_amenagement = Column(Unicode(120))
    date_arrete = Column(Date)
    date_decision = Column(Date)
    annee_debut_amenagement = Column(Unicode(4))
    annee_echeance_amenagement = Column(Unicode(4))
    date_fin_applicabilite = Column(Date)
    cadre_legal_gestion = Column(Unicode(3))
    surface_cadastrale = Column(Numeric(15, 4))
    surface_retenue_gestion = Column(Numeric(15, 4))
    surface_boisee_debug_amgt = Column(Numeric(15, 4))
    surface_sylviculture = Column(Numeric(15, 4))
    surface_hors_sylviculture = Column(Numeric(15, 4))
    date_deliberation = Column(Date)
    date_envoi_proprietaire = Column(Date)
    date_envoi_etat_pr_appro = Column(Date)
    date_envoi_approuve = Column(Date)
    libelle_recherche = Column(Unicode(128))
    id_pdf_fiche_synthese = Column(Unicode(8))
    id_fiche_synthese_public = Column(Unicode(8))
    id_pdf_amenagement = Column(Unicode(8))
    id_pdf_amenagement_public = Column(Unicode(8))
    id_pdf_envoi_proprietaire = Column(Unicode(8))
    id_pdf_deliberation_propri0 = Column(Unicode(8))
    id_pdf_arrete_amenagement = Column(Unicode(8))
    id_pdf_carto1 = Column(Unicode(8))
    id_pdf_carto2 = Column(Unicode(8))
    id_pdf_carto3 = Column(Unicode(8))
    id_pdf_carto4 = Column(Unicode(8))
    id_pdf_carto5 = Column(Unicode(8))
    date_debut_applicabilite = Column(Date)
    surface_a_ouvrir = Column(Numeric(15, 4))
    surface_a_acquerir = Column(Numeric(15, 4))
    surface_a_terminer = Column(Numeric(15, 4))
    surface_foret_protection = Column(Numeric(15, 4))
    surface_coeur_parc_nat = Column(Numeric(15, 4))
    surface_reserv_nat_reg = Column(Numeric(15, 4))
    surface_rbi = Column(Numeric(15, 4))
    surface_rbd = Column(Numeric(15, 4))
    surface_arrete_protection = Column(Numeric(15, 4))
    surface_site_classe = Column(Numeric(15, 4))
    surface_monu_hist_ins = Column(Numeric(15, 4))
    surface_monu_hist_cla = Column(Numeric(15, 4))
    surface_perimetres_captages = Column(Numeric(15, 4))
    surface_adhesion_parc_nat = Column(Numeric(15, 4))
    surface_parc_nat_reg = Column(Numeric(15, 4))
    surface_charte_forestiere = Column(Numeric(15, 4))
    surface_natura2000_habitats = Column(Numeric(15, 4))
    surface_natura2000_oiseaux = Column(Numeric(15, 4))
    surface_znieff_type_i = Column(Numeric(15, 4))
    surface_znieff_type_ii = Column(Numeric(15, 4))
    surface_unites_conservation = Column(Numeric(15, 4))
    surface_plan_prev_risq_nat = Column(Numeric(15, 4))
    surface_plan_prev_risq_inc = Column(Numeric(15, 4))
    surface_zone_ret_eau = Column(Numeric(15, 4))
    surface_reserv_nat_chasse = Column(Numeric(15, 4))
    surface_pastoralisme = Column(Numeric(15, 4))
    surface_prob_sanit_graves = Column(Numeric(15, 4))
    surface_desiquilib_fau_flo = Column(Numeric(15, 4))
    surface_incendies = Column(Numeric(15, 4))
    surface_pb_fonciers = Column(Numeric(15, 4))
    surface_pres_essence_vs_climat = Column(Numeric(15, 4))


    @declared_attr
    def __table_args__(cls):
        return (
            ForeignKeyConstraint(['departement_situation', 'departement_situation_suppl'],
                                 [Departement.dep, Departement.dep]),
            cls.get_schema()
        )

    # declared attributes

    @declared_attr
    def departement_situation_insee(self):
        return relationship(Departement,
                            primaryjoin=Departement.dep==self.departement_situation)

    @declared_attr
    def departement_situation_suppl_insee(self):
        return relationship(Departement,
                            primaryjoin=Departement.dep==self.departement_situation_suppl)
    
    #properties
    
    @property
    def value(self):
        return self.id_nat_amgt

    @property
    def token(self):
        return self.id_nat_amgt

    @property
    def title(self):
        return self.nom_usage_amenagement

    @classmethod
    def get(cls, code, session=PARENT_SESSION):
        if isinstance(code, str):
            code = code.split(',')
        session = get_user_session(session)
        return session.query(PlanningData).filter(PlanningData.id_nat_amgt.in_(code))

    @classmethod
    def find(cls, query, session=PARENT_SESSION, ignore_disabled=False):
        if isinstance(query, dict):
            id_nat_amgt = query.get('id_nat_amgt')
            label = query.get('label')
        else:
            query = query.strip()
            if REG_ID_NAT_AMGT.match(query.upper()):
                id_nat_amgt = query.upper()
                label = None
            else:
                id_nat_amgt = None
                label = query
        session = get_user_session(session)
        if id_nat_amgt:
            return PlanningData.find_by_id(id_nat_amgt, session, ignore_disabled)
        elif label:
            return PlanningData.find_by_label(label, session, ignore_disabled)

    @classmethod
    def find_by_id(cls, id_nat_amgt, session=PARENT_SESSION, ignore_disabled=False):
        if isinstance(id_nat_amgt, (list, tuple, set)):
            params = PlanningData.id_nat_amgt.in_(id_nat_amgt)
        else:
            params = PlanningData.id_nat_amgt == id_nat_amgt
        session = get_user_session(session)
        return session.query(PlanningData).filter(params)

    @classmethod
    def find_by_label(cls, query, session=PARENT_SESSION, ignore_disabled=False):
        query = '%{}%'.format(translate_string(query, keep_chars="_-.'").upper()) \
            .replace('-', ' ') \
            .replace('_', '|_')
        params = [PlanningData.libelle_recherche.like(query, escape='|')]
        if ignore_disabled:
            today = date.today()
            params.append(PlanningData.date_debut_applicabilite <= today)
            params.append(PlanningData.date_fin_applicabilite >= today)
        session = get_user_session(session)
        return session.query(PlanningData) \
                      .filter(and_(*params))


class PlanningForet(DynamicSchemaMixin, Base):
    """Model class for v_foret_par_amenagement table"""

    __tablename__ = 'v_foret_par_amenagement'
    __schema__ = PARENT_SCHEMA

    @declared_attr
    def __table_args__(cls):
        return (
            PrimaryKeyConstraint('identifiant_nat_amenagement', 'id_nat_foret'),
            cls.get_schema()
        )

    id_nat_amgt = Column('identifiant_nat_amenagement', Unicode(8),
                         ForeignKey(PlanningData.id_nat_amgt))
    id_nat_foret = Column(Unicode(7), ForeignKey(Foret.id_nat_frt))
    surface_cadastrale = Column(Numeric(15, 4))
    surface_retenue_gestion = Column(Numeric(15, 4))

    planning = relation(PlanningData, backref='forets')
    foret = relation(Foret, backref='plannings')


PlanningForet.info_foret = relation(InformationForet,
                                    foreign_keys=PlanningForet.id_nat_foret,
                                    primaryjoin=PlanningForet.id_nat_foret==InformationForet.id_nat_frt,
                                    uselist=False)


# Table RDF_PLANNING_CUSTOM_INFO

@implementer(IInformationPlanning)
class InformationPlanning(DynamicSchemaMixin, Base):
    """Model class for V_PLANNING_CUSTOM_INFO"""

    __tablename__ = 'v_amenagement_custom_info'
    __schema__ = PARENT_SCHEMA

    id_nat_amgt = Column(Unicode(8), ForeignKey(PlanningData.id_nat_amgt), primary_key=True)
    libelle = Column(Unicode(255))
    visible = Column(Boolean(), nullable=False, default=True)
    header = Column(Text())

    planning = relation(PlanningData, backref='informations')


# Table v_fichiers_modif_amenagement

@implementer(IModificationPlanning)
class ModificationPlanning(DynamicSchemaMixin, Base):
    """Model class for v_fichiers_modif_amenagement"""

    __tablename__ = 'v_fichiers_modif_amenagement'
    __schema__ = PARENT_SCHEMA

    id_nat_amgt = Column('identifiant_nat_amenagement', Unicode(8),
                         ForeignKey(PlanningData.id_nat_amgt))
    numero_modification = Column(Integer)
    type_fichier = Column(Unicode(60))
    id_fichier = Column(Unicode(8), primary_key=True)


ModificationPlanning.planning = relation(PlanningData, backref='modifs',
                                         order_by=(ModificationPlanning.numero_modification,
                                                   ModificationPlanning.type_fichier))
