#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from onf_website.reference.location.interfaces.territory import ITerritoryTable, ITerritory
from pyams_i18n.interfaces import II18n
from zope.component.interfaces import ISite
from zope.lifecycleevent.interfaces import IObjectAddedEvent

# import packages
from pyams_content.reference import ReferenceTable, ReferenceInfo
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyramid.events import subscriber
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleVocabulary, SimpleTerm


@implementer(ITerritoryTable)
class TerritoryTable(ReferenceTable):
    """Territories table"""


@subscriber(IObjectAddedEvent, context_selector=ITerritoryTable)
def handle_added_territory_table(event):
    """Handle new territory table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, ITerritoryTable)


@implementer(ITerritory)
class Territory(ReferenceInfo):
    """Territory record"""

    internal_code = FieldProperty(ITerritory['internal_code'])
    regions = FieldProperty(ITerritory['regions'])


@vocabulary_config(name='ONF territories')
class TerritoryVocabulary(SimpleVocabulary):
    """Territories vocabulary"""

    def __init__(self, context=None):
        table = query_utility(ITerritoryTable)
        if table is not None:
            request = check_request()
            terms = [SimpleTerm(v,
                                token=v.__name__,
                                title='{0} - {1}'.format(v.internal_code,
                                                         II18n(v).query_attribute('title', request=request)))
                     for v in table.values()]
        else:
            terms = []
        super(TerritoryVocabulary, self).__init__(terms)
