#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

import json

from z3c.form.browser.widget import HTMLInputWidget
from z3c.form.converter import BaseDataConverter
from z3c.form.interfaces import IDataConverter, IFieldWidget
from z3c.form.widget import FieldWidget, Widget
from zope.interface import implementer_only

from onf_website.reference.insee.model import Commune, Departement
from onf_website.reference.insee.schema import ICommuneField, ICommunesListField, \
    IDepartmentField, IDepartmentsListField
from onf_website.reference.insee.zmi.interfaces import ICommuneWidget, ICommunesListWidget, \
    IDepartmentWidget, IDepartmentsListWidget
from pyams_form.interfaces.form import IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_utils.adapter import adapter_config


#
# Department field
#

@adapter_config(context=(IDepartmentField, IDepartmentWidget), provides=IDataConverter)
class DepartmentDataConverter(BaseDataConverter):
    """Department data converter"""

    def toWidgetValue(self, value):
        if not value:
            return None
        return Departement.find_by_insee_code(value).first()

    def toFieldValue(self, value):
        return value or None


@widgettemplate_config(mode='input', template='templates/department-input.pt', layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/department-display.pt',
                       layer=IFormLayer)
@implementer_only(IDepartmentWidget)
class DepartmentWidget(HTMLInputWidget, Widget):
    """Department widget"""

    @property
    def value_map(self):
        if not self.value:
            return ''
        if isinstance(self.value, str):
            self.value = Departement.find_by_insee_code(self.value).first()
        return json.dumps({self.value.token: self.value.title})


@adapter_config(context=(IDepartmentField, IFormLayer), provides=IFieldWidget)
def DepartmentFieldWidget(field, request):
    """Department field widget factory"""
    return FieldWidget(field, DepartmentWidget(request))


#
# Departments list field
#

@adapter_config(context=(IDepartmentsListField, IDepartmentsListWidget), provides=IDataConverter)
class DepartmentsListDataConverter(BaseDataConverter):
    """Departments list data converter"""

    def toWidgetValue(self, value):
        if not value:
            return ()
        return Departement.find_by_insee_code(value).all()

    def toFieldValue(self, value):
        if not value:
            return []
        return value.split('|')


@widgettemplate_config(mode='input', template='templates/departments-list-input.pt',
                       layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/departments-list-display.pt',
                       layer=IFormLayer)
@implementer_only(IDepartmentsListWidget)
class DepartmentsListWidget(HTMLInputWidget, Widget):
    """Departments list widget"""

    @property
    def values(self):
        return '|'.join((department.token for department in self.value or ()))

    @property
    def values_map(self):
        result = {}
        [result.update({value.token: value.title}) for value in self.value or ()]
        return json.dumps(result)


@adapter_config(context=(IDepartmentsListField, IFormLayer), provides=IFieldWidget)
def DepartmentsListFieldWidget(field, request):
    """Departments list field widget factory"""
    return FieldWidget(field, DepartmentsListWidget(request))


#
# Commune field
#

@adapter_config(context=(ICommuneField, ICommuneWidget), provides=IDataConverter)
class CommuneDataConverter(BaseDataConverter):
    """Commune data converter"""

    def toWidgetValue(self, value):
        if not value:
            return None
        return Commune.find_by_insee_code(value).first()

    def toFieldValue(self, value):
        return value or None


@widgettemplate_config(mode='input', template='templates/commune-input.pt', layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/commune-display.pt', layer=IFormLayer)
@implementer_only(ICommuneWidget)
class CommuneWidget(HTMLInputWidget, Widget):
    """Commune widget"""

    @property
    def value_map(self):
        if not self.value:
            return ''
        if isinstance(self.value, str):
            self.value = Commune.find_by_insee_code(self.value).first()
        return json.dumps({self.value.token: self.value.title})


@adapter_config(context=(ICommuneField, IFormLayer), provides=IFieldWidget)
def CommuneFieldWidget(field, request):
    """Commune field widget factory"""
    return FieldWidget(field, CommuneWidget(request))


#
# Communes list field
#

@adapter_config(context=(ICommunesListField, ICommunesListWidget), provides=IDataConverter)
class CommunesListDataConverter(BaseDataConverter):
    """Communes list data converter"""

    def toWidgetValue(self, value):
        if not value:
            return ()
        return Commune.find_by_insee_code(value).all()

    def toFieldValue(self, value):
        if not value:
            return []
        return value.split('|')


@widgettemplate_config(mode='input', template='templates/communes-list-input.pt', layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/communes-list-display.pt', layer=IFormLayer)
@implementer_only(ICommunesListWidget)
class CommunesListWidget(HTMLInputWidget, Widget):
    """Communes list widget"""

    @property
    def values(self):
        return '|'.join((commune.token for commune in self.value or ()))

    @property
    def values_map(self):
        result = {}
        [result.update({value.token: value.title}) for value in self.value or ()]
        return json.dumps(result)


@adapter_config(context=(ICommunesListField, IFormLayer), provides=IFieldWidget)
def CommunesListFieldWidget(field, request):
    """Communes list field widget factory"""
    return FieldWidget(field, CommunesListWidget(request))
