#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces

# import packages
from zope.interface import Interface
from zope.schema import TextLine, Object, List

from onf_website import _


class ICodeLibelle(Interface):
    """Code/label record interface"""
    code = TextLine(title=_("Code"))
    libelle = TextLine(title=_("Label"))


class ICodeCDC(ICodeLibelle):
    """CDC code record interface"""


class ICodeTNCC (ICodeLibelle):
    """TNCC code record interface"""


class ICodeTypCT(Interface):
    """Type CT code record interface"""


class ICodeChefLieu(Interface):
    """Cheflieu code record interface"""


class IPays(Interface):
    """Country record interface"""
    cog = TextLine(title=_("Code"))
    actual = TextLine(title=_("Actual country code"))
    capay = TextLine(title=_("Previous country code"))
    crpay = TextLine(title=_("New group country code"))
    ani = TextLine(title=_("Independance year"))
    libcog = TextLine(title=_("Short name"))
    libenr = TextLine(title=_("Rich name"))
    ancnom = TextLine(title=_("Previous name"))
    code_iso = TextLine(title=_("ISO code"))


class IRegionBase(Interface):
    """Region record interface"""
    region = TextLine(title=_("Code"))
    cheflieu = TextLine(title=_("Chef-lieu commune's code"))
    tncc = TextLine(title=_("TNCC code"))
    ncc = TextLine(title=_("Uppercase label"))
    nccenr = TextLine(title=_("Lowercase label"))


class IDepartementBase(Interface):
    """Departement record interface"""
    region = TextLine(title=_("Region code"))
    dep = TextLine(title=_("Department code"))
    cheflieu = TextLine(title=_("Chef-lieu commune's code"))
    tncc = TextLine(title=_("TNCC code"))
    ncc = TextLine(title=_("Uppercase label"))
    nccenr = TextLine(title=_("Lowercase label"))


class IDepartementModel(Interface):
    """Departement class model interface"""


class IArrondissementBase(Interface):
    """Arrondissement record interface"""
    region = TextLine(title=_("Region code"))
    dep = TextLine(title=_("Department code"))
    ar = TextLine(title=_("Arrondissement code"))
    cheflieu = TextLine(title=_("Chef-lieu commune's code"))
    tncc = TextLine(title=_("TNCC code"))
    artmaj = TextLine(title=_("Uppercase adjective"))
    ncc = TextLine(title=_("Uppercase label"))
    artmin = TextLine(title=_("Lowercase adjective"))
    nccenr = TextLine(title=_("Lowercase label"))


class ICantonBase(Interface):
    """Canton record interface"""
    region = TextLine(title=_("Region code"))
    dep = TextLine(title=_("Department code"))
    ar = TextLine(title=_("Arrondissement code"))
    canton = TextLine(title=_("Canton code"))
    typct = TextLine(title=_("CT type code"))
    cheflieu = TextLine(title=_("Chef-lieu commune's code"))
    tncc = TextLine(title=_("TNCC code"))
    artmaj = TextLine(title=_("Uppercase adjective"))
    ncc = TextLine(title=_("Uppercase label"))
    artmin = TextLine(title=_("Lowercase adjective"))
    nccenr = TextLine(title=_("Lowercase label"))


class ICommuneBase(Interface):
    """Commune record base interface"""
    cdc = TextLine(title=_("CDC"))
    cheflieu = TextLine(title=_("Chef lieu"))
    reg = TextLine(title=_("Region code"))
    dep = TextLine(title=_("Department code"))
    com = TextLine(title=_("Commune code"))
    ar = TextLine(title=_("Arrondissement code"))
    ct = TextLine(title=_("Canton code"))
    tncc = TextLine(title=_("TNCC code"))
    artmaj = TextLine(title=_("Uppercase adjective"))
    ncc = TextLine(title=_("Uppercase label"))
    artmin = TextLine(title=_("Lowercase adjective"))
    nccenr = TextLine(title=_("Lowercase label"))


class ICommuneModel(Interface):
    """Commune class model interface"""


class ICodePostalBase(Interface):
    """Code postal record interface"""
    code_insee = TextLine(title=_("Commune INSEE code"))
    reserve = TextLine(title=_("Reserved"))
    libelle = TextLine(title=_("Label"))
    code_postal = TextLine(title=_("Postal code"))
    distributeur = TextLine(title=_("Distribution desk"))
    libelle_cedex = TextLine(title=_("Cedex label"))
    filler = TextLine(title=_("Filler"))


# References interfaces

class IRegionRefs(Interface):
    """External links for regions records"""
    ref_cheflieu = Object(title=_("Commune reference"), schema=Interface)
    ref_tncc = Object(title=_("TNCC reference"), schema=Interface)
    ref_departements = List(value_type=Object(title=_("Regions"), schema=Interface))


class IDepartementRefs(Interface):
    """External links for departements records"""
    ref_region = Object(title=_("Region reference"), schema=Interface)
    ref_cheflieu = Object(title=_("Commune reference"), schema=Interface)
    ref_tncc = Object(title=_("TNCC reference"), schema=Interface)


class IArrondissementRefs(Interface):
    """External links for arrondissements records"""
    ref_region = Object(title=_("Region reference"), schema=Interface)
    ref_departement = Object(title=_("Departement reference"), schema=Interface)
    ref_cheflieu = Object(title=_("Commune reference"), schema=Interface)
    ref_tncc = Object(title=_("TNCC reference"), schema=Interface)


class ICantonRefs(Interface):
    """External links for cantons records"""
    ref_region = Object(title=_("Region reference"), schema=Interface)
    ref_departement = Object(title=_("Departement reference"), schema=Interface)
    ref_arrondissement = Object(title=_("Arrondissement reference"), schema=Interface)
    ref_typect = Object(title=_("CT reference"), schema=Interface)
    ref_cheflieu = Object(title=_("Commune reference"), schema=Interface)
    ref_tncc = Object(title=_("TNCC reference"), schema=Interface)


class ICommuneRefs(Interface):
    """External links for commune records"""
    ref_cdc = Object(title=_("CDC type reference"), schema=Interface)
    ref_codecheflieu = Object(title=_("Cheflieu code reference"), schema=Interface)
    ref_region = Object(title=_("Region reference"), schema=Interface)
    ref_departement = Object(title=_("Departement reference"), schema=Interface)
    ref_arrondissement = Object(title=_("Arrondissement reference"), schema=Interface)
    ref_tncc = Object(title=_("TNCC reference"), schema=Interface)
    ref_codespostaux = List(value_type=Object(title=_("Postal codes"), schema=Interface))


class ICodePostalRefs(Interface):
    """External links for postal codes"""
    ref_communes = List(value_type=Object(title=_("Communes references"), schema=Interface))


# Full interfaces

class IRegion(IRegionBase, IRegionRefs):
    """Region full interface"""


class IDepartement(IDepartementBase, IDepartementRefs):
    """Departement full interface"""


class IArrondissement(IArrondissementBase, IArrondissementRefs):
    """Arrondissement full interface"""


class ICanton(ICantonBase, ICantonRefs):
    """Canton full interface"""


class ICommune(ICommuneBase, ICommuneRefs):
    """Commune record interface"""


class ICodePostal(ICodePostalBase, ICodePostalRefs):
    """Postal codes record interface"""


# Updated schema attributes

IRegion['ref_cheflieu'].schema = ICommune
IRegion['ref_tncc'].schema = ICodeTNCC
IRegion['ref_departements'].value_type.schema = IDepartement

IDepartement['ref_region'].schema = IRegion
IDepartement['ref_cheflieu'].schema = ICommune
IDepartement['ref_tncc'].schema = ICodeTNCC

IArrondissement['ref_region'].schema = IRegion
IArrondissement['ref_departement'].schema = IDepartement
IArrondissement['ref_cheflieu'].schema = ICommune
IArrondissement['ref_tncc'].schema = ICodeTNCC

ICanton['ref_region'].schema = IRegion
ICanton['ref_departement'].schema = IDepartement
ICanton['ref_arrondissement'].schema = IArrondissement
ICanton['ref_typect'].schema = ICodeTypCT
ICanton['ref_cheflieu'].schema = ICommune
ICanton['ref_tncc'].schema = ICodeTNCC

ICommune['ref_cdc'].schema = ICodeCDC
ICommune['ref_codecheflieu'].schema = ICodeChefLieu
ICommune['ref_region'].schema = IRegion
ICommune['ref_departement'].schema = IDepartement
ICommune['ref_arrondissement'].schema = IArrondissement
ICommune['ref_tncc'].schema = ICodeTNCC
ICommune['ref_codespostaux'].value_type.schema = ICodePostal

ICodePostal['ref_communes'].value_type.schema = ICommune
