#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL). A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS Insee API Module"""

__docformat__ = 'restructuredtext'

from cornice import Service
from cornice.validators import colander_validator

from pyramid.httpexceptions import  HTTPOk

from onf_website.reference.insee.api.interfaces import REST_INSEE_ROUTE
from onf_website.reference.insee.model import Commune
from onf_website.reference.insee.api.schema import InseeGetterRequest, InseeGetterResponse

from pyams_security.rest import check_cors_origin, set_cors_headers
from pyams_utils.rest import STATUS, rest_responses




def format_commune(commune):
    """Helper function to format commune details."""
    return {
        "insee": commune.code,
        "artmaj": commune.artmaj,
        "label": commune.nccenr
    }


#
# API Insee
#

insee_service = Service(name=REST_INSEE_ROUTE,
                        pyramid_route=REST_INSEE_ROUTE,
                        description="ONF website API for Insee")


@insee_service.options(validators=(check_cors_origin, set_cors_headers))
def insee_options(request):
    return ''


insee_responses = rest_responses.copy()
insee_responses[HTTPOk.code] = InseeGetterResponse(description="Insee search results")


@insee_service.get(schema=InseeGetterRequest(),
                   validators=(check_cors_origin,colander_validator,set_cors_headers),
                   response_schemas=insee_responses)
def insee_getter(request) :
    """Insee settings retrieval"""
    insee_code = request.matchdict.get('insee_code')
    if not insee_code or len(insee_code) < 2 :
        return []
    commune = Commune.find_by_insee_code(insee_code).first()
    if not commune:
        return []
    return {
        'status': STATUS.SUCCESS.value,
        'commune': format_commune(commune)
    }
