#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from z3c.form.browser.widget import HTMLFormElement, HTMLInputWidget
from z3c.form.converter import BaseDataConverter
from z3c.form.interfaces import DISPLAY_MODE, IDataConverter, IFieldWidget, INPUT_MODE, NO_VALUE
from z3c.form.widget import FieldWidget, MultiWidget, Widget
from zope.interface import implementer_only
from zope.schema.vocabulary import getVocabularyRegistry

from onf_website.reference.forest.interfaces import FOREST_OWNERS_CATEGORIES
from onf_website.reference.forest.model.foret import Foret
from onf_website.reference.forest.schema import IForestField, IForestsListField
from onf_website.reference.forest.zmi.interfaces import IForestWidget, IForestsListWidget
from pyams_form.interfaces.form import IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_utils.adapter import adapter_config


__docformat__ = 'restructuredtext'


#
# Forest field
#

@adapter_config(context=(IForestField, IForestWidget), provides=IDataConverter)
class ForestDataConverter(BaseDataConverter):
    """Forest data converter"""

    def toWidgetValue(self, value):
        if not value:
            return None
        return Foret.find_by_id(value).first()

    def toFieldValue(self, value):
        return value


@widgettemplate_config(mode='input', template='templates/forest-input.pt', layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/forest-display.pt', layer=IFormLayer)
@implementer_only(IForestWidget)
class ForestWidget(HTMLInputWidget, Widget):
    """Forest widget"""

    @property
    def value_map(self):
        if not self.value:
            return ''
        return json.dumps({self.value.token: self.value.title})


@adapter_config(context=(IForestField, IFormLayer), provides=IFieldWidget)
def ForestFieldWidget(field, request):
    """Forest field widget factory"""
    return FieldWidget(field, ForestWidget(request))


#
# Forests list field
#

@adapter_config(context=(IForestsListField, IForestsListWidget), provides=IDataConverter)
class ForestsListDataConverter(BaseDataConverter):
    """Forests list data converter"""

    def toWidgetValue(self, value):
        if not value:
            return ()
        return Foret.find_by_id(value).all()

    def toFieldValue(self, value):
        if not value:
            return []
        return value.split('|')


@widgettemplate_config(mode='input', template='templates/forests-list-input.pt', layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/forests-list-display.pt', layer=IFormLayer)
@implementer_only(IForestsListWidget)
class ForestsListWidget(HTMLInputWidget, Widget):
    """Forests list widget"""

    @property
    def values(self):
        return '|'.join((foret.token for foret in self.value or ()))

    @property
    def values_map(self):
        result = {}
        [result.update({value.token: value.title}) for value in self.value or ()]
        return json.dumps(result)


@adapter_config(context=(IForestsListField, IFormLayer), provides=IFieldWidget)
def ForestsListFieldWidget(field, request):
    """Forests list field widget factory"""
    return FieldWidget(field, ForestsListWidget(request))


@widgettemplate_config(mode=INPUT_MODE,
                       template='templates/displays-input.pt', layer=IFormLayer)
@widgettemplate_config(mode=DISPLAY_MODE,
                       template='templates/displays-display.pt', layer=IFormLayer)
class ForestTableDisplaysSelectWidget(HTMLFormElement, MultiWidget):
    """Forest table displays select widget"""

    @property
    def items(self):
        displays = self.field.bind(self.context).get(self.context) or {}
        vocabulary = getVocabularyRegistry().get(self.context, FOREST_OWNERS_CATEGORIES)
        for value, term in sorted(vocabulary.by_value.items(),
                                  key=lambda x: x[1].value):
            yield term, displays.get(value)

    def updateWidgets(self):
        pass

    def extract(self, default=NO_VALUE):
        if self.request.get('{}_marker'.format(self.name)) is None:
            return default
        result = []
        vocabulary = getVocabularyRegistry().get(self.context, FOREST_OWNERS_CATEGORIES)
        for value, term in vocabulary.by_value.items():
            result.append([
                (value,),
                ('true' if self.request.get('{}_{}'.format(self.name, value))
                 else 'false',)
            ])
        return result


def ForestTableDisplaysSelectWidgetFactory(field, request):
    """Forest table displays select widget factory"""
    return FieldWidget(field, ForestTableDisplaysSelectWidget(request))
