#
# Copyright (c) 2012 Thierry Florac <tflorac AT onf.fr>
# All Rights Reserved.
#


from sqlalchemy import Column, Unicode
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.schema import PrimaryKeyConstraint
from sqlalchemy.sql import and_
from zope.schema.vocabulary import SimpleVocabulary, SimpleTerm
from onf_website.reference.forest.interfaces import FOREST_OWNERS_CATEGORIES
from onf_website.reference.forest.model import PARENT_SCHEMA, PARENT_SESSION
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin
from pyams_utils.vocabulary import vocabulary_config


from onf_website import _


class Code(DynamicSchemaMixin, Base):
    """Model class for IFN_CODE table"""

    __tablename__ = 'rdf_typo'
    __schema__ = PARENT_SCHEMA

    @declared_attr
    def __table_args__(cls):
        return (
            PrimaryKeyConstraint('unite', 'code'),
            cls.get_schema()
        )

    unite = Column(Unicode)
    code = Column(Unicode)
    libelle = Column(Unicode)


def get_libelle(unite, code, session=PARENT_SESSION):
    """Extract label for given code and unit"""
    session = get_user_session(session)
    try:
        return session.query(Code).filter(and_(Code.unite == unite,
                                               Code.code == code)).first().libelle
    except:
        return _("< missing reference label >")


@vocabulary_config(name=FOREST_OWNERS_CATEGORIES)
class ForestOwnersCategoryVocabulary(SimpleVocabulary):
    """Forest owners vocabulary"""

    def __init__(self, context=None):
        session = get_user_session(PARENT_SESSION)
        terms = [
            SimpleTerm(code.code, title=code.libelle)
            for code in session.query(Code)
                .filter(Code.unite == 'CATEGORIE_PROPRIETAIRE')
                .order_by(Code.libelle)
        ]
        super().__init__(terms)
