#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""
from colander import Float, MappingSchema, SchemaNode, SequenceSchema, String, drop

from pyams_content.component.illustration.api.schema import IllustrationInfo
from pyams_content.component.theme.api.schema import ThesaurusTermsList
from pyams_content.shared.alert.api.schema import AlertTypesList
from pyams_content.shared.common.api.schema import BaseContentInfoSearchQuery, DataTypesList
from pyams_utils.rest import BaseResponseSchema, StringListSchema


class ForestSchema(MappingSchema):
    """Forest item schema"""
    id_nat_frt = SchemaNode(String(),
                            description="Forest national ID")
    libelle_usage = SchemaNode(String(),
                               description="Forest usage label")
    libelle_strict = SchemaNode(String(),
                                description="Forest strict label")
    illustration = IllustrationInfo(description="Illustration information",
                                    missing=drop)


class ForestsList(SequenceSchema):
    """Forests IDs list"""
    forest = ForestSchema()


class ForetMapperConfigResultSchema(BaseResponseSchema):
    """ForetMapper configuration schema"""
    forests_route = SchemaNode(String(),
                               description="Forests public API route path")
    alerts_route = SchemaNode(String(),
                              description="Alerts public API route path")
    active_forests = ForestsList(description="List of forests IDs which are active for ForetMapper")
    alert_types = AlertTypesList(description="List of alert types",
                                 missing=drop)
    activity_types = DataTypesList(description="List of activity types",
                                   missing=drop)
    activity_adaptations = ThesaurusTermsList(dsceription="List of adaptations themes",
                                              missing=drop)


class ForestsQuerySchema(MappingSchema):
    """Forests query"""
    longitude = SchemaNode(Float(),
                           description="Origin position longitude in WGS84 coordinates")
    latitude = SchemaNode(Float(),
                          description="Origin position latitude in WGS84 coordinates")
    distance = SchemaNode(Float(),
                          description="Distance from location, in kilometers; default=50",
                          missing=50.)
    forestTypes = SchemaNode(String(),
                             description="Forests owners categories, as codified in the RDF; "
                                         "default='1' for domain forests",
                             missing="1")


class ForestResult(MappingSchema):
    """Base forest result"""
    id_nat_frt = SchemaNode(String(),
                            title="Forest national ID")
    libelle_usage = SchemaNode(String(),
                               title="Forest public label")
    libelle_strict = SchemaNode(String(),
                                title="Forest strict label")
    distance = SchemaNode(Float(),
                          title="Forest distance from origin, in kilometers",
                          missing=drop)


class ForestsResultsList(SequenceSchema):
    """Forests results list"""
    forest = ForestResult()


class ForestsQueryResponse(BaseResponseSchema):
    """Forests response schema"""
    results = ForestsResultsList(description="Forests search results, ordered by distance from "
                                             "specified location",
                                 missing=drop)


class ForestMapperConfigRequest(MappingSchema):
    """ForetMapper configuration request schema"""
    querystring = BaseContentInfoSearchQuery()


class ForetMapperConfigResponse(MappingSchema):
    """ForetMapper configuration getter response"""
    body = ForetMapperConfigResultSchema()


class ForestsGetterRequest(MappingSchema):
    """Forests getter request"""
    querystring = ForestsQuerySchema()


class ForestsGetterResponse(MappingSchema):
    """Forests getter response"""
    body = ForestsQueryResponse()
