#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from datetime import datetime

from persistent import Persistent
from pyramid.events import subscriber
from sqlalchemy.sql import and_, or_
from zope.component.interfaces import ISite
from zope.container.contained import Contained
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.reference.forest.interfaces import IForestTable
from onf_website.reference.forest.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.forest.model.foret import Foret, InformationForet, ProprietaireForet
from pyams_alchemy.engine import get_user_session
from pyams_i18n.interfaces import II18nManager
from pyams_utils.timezone import tztime
from pyams_utils.traversing import get_parent


@implementer(IForestTable, II18nManager)
class ForestTable(Persistent, Contained):
    """Forest table manager"""

    title = FieldProperty(IForestTable['title'])
    short_name = FieldProperty(IForestTable['short_name'])

    languages = FieldProperty(II18nManager['languages'])

    forest_displays = FieldProperty(IForestTable['forest_displays'])
    hunting_displays = FieldProperty(IForestTable['hunting_displays'])

    @property
    def visible_owners_ids(self):
        return [
            key
            for (key, value) in self.forest_displays.items()
            if value
        ]

    def get_published_forests(self):
        session = get_user_session(RDF_SESSION)
        now = tztime(datetime.utcnow())
        yield from session.query(Foret, InformationForet) \
            .join(ProprietaireForet, Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
            .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .filter(and_(Foret.date_debut_validite <= now,
                         Foret.date_fin_validite >= now,
                         ProprietaireForet.categorie.in_(self.visible_owners_ids),
                         or_(InformationForet.visible == None,
                             InformationForet.visible == True))) \
            .order_by(Foret.libelle_usage)

    @property
    def visible_hunting_calendars_owners_ids(self):
        return [
            key
            for (key, value) in self.hunting_displays.items()
            if value
        ]
    
    def get_published_hunting_calendars(self):
        session = get_user_session(RDF_SESSION)
        now = tztime(datetime.utcnow())
        yield from session.query(Foret, InformationForet) \
            .join(ProprietaireForet, Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
            .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .filter(and_(Foret.date_debut_validite <= now,
                         Foret.date_fin_validite >= now,
                         ProprietaireForet.categorie.in_(self.visible_hunting_calendars_owners_ids),
                         or_(InformationForet.visible == None,
                             InformationForet.visible == True))) \
            .order_by(Foret.libelle_usage)


@subscriber(IObjectAddedEvent, context_selector=IForestTable)
def handle_added_forest_table(event):
    """Handle new forest table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IForestTable)
