# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from sqlalchemy import Column, Unicode, and_

from onf_website.reference.orga.model import Structure
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin


PARENT_SCHEMA = 'annuaire'
PARENT_SESSION = 'ANNUAIRE'


class Agent(DynamicSchemaMixin, Base):
    """Model class for agent table"""
    
    __tablename__ = 'agent'
    __schema__ = PARENT_SCHEMA
    
    nomat = Column(Unicode(5), primary_key=True)
    nom = Column(Unicode(65))
    prenom = Column(Unicode(25))
    no_poste = Column(Unicode(40))
    telephone = Column(Unicode(20))
    portable = Column(Unicode(20))
    mail = Column(Unicode(100))

    @classmethod
    def get(cls, code, session=PARENT_SESSION):
        if isinstance(code, str):
            code = code.split(',')
        session = get_user_session(session)
        return session.query(cls).filter(cls.nomat.in_(code))
    
    @classmethod
    def find_by_insee_code(cls, insee_code, session=PARENT_SESSION):
        """Get UT and contacts for given commune"""
        session = get_user_session(session)
        commune = session.query(CommuneDac).filter(CommuneDac.code_commune == insee_code).first()
        if not commune :
            return None

        data = session.query(Structure, UtDac) \
            .join(Structure, Structure.code_sign == UtDac.no_ut) \
            .filter(Structure.code_sign == commune.no_ut).first()
        if data:
            structure, ut = data
            agence = session.query(Structure).filter(Structure.code_sign == structure.code_sign[:4]).first()
            dt = session.query(Structure).filter(Structure.code_sign == structure.code_sign[:2]).first()
            if ut.nomat_agent_interim:
                data = session.query(Poste, Agent) \
                        .join(Poste, Poste.no == Agent.no_poste) \
                        .filter(Agent.nomat == ut.nomat_agent_interim).first()
            elif ut.no_poste_interim:
                data = session.query(Poste, Agent) \
                        .join(Poste, Poste.no == Agent.no_poste) \
                        .filter(Poste.no == ut.no_poste_interim).first()
            else:
                data = session.query(Poste, Agent) \
                        .join(Poste, Poste.no == Agent.no_poste) \
                        .filter(Poste.no == ut.no_poste_dir).first()
            if data:
                poste_resp, agent_resp = data
            else:
                poste_resp, agent_resp = Poste(), Agent()
        else:
            structure, poste_resp, agent_resp, ut, agence, dt = \
                Structure(), Poste(), Agent(), UtDac(), Structure(), Structure()

        if commune.nomat_agent:
            data = session.query(Poste, Agent) \
                    .join(Poste, Poste.no == Agent.no_poste) \
                    .filter(Agent.nomat == commune.nomat_agent).first()
        elif commune.no_poste:
            data = session.query(Poste, Agent) \
                    .join(Poste, Poste.no == Agent.no_poste) \
                    .filter(Poste.no == commune.no_poste).first()
        elif ut.nomat_agent:
            data = session.query(Poste, Agent) \
                    .join(Poste, Poste.no == Agent.no_poste) \
                    .filter(Agent.nomat == ut.nomat_agent).first()
        elif ut.no_poste:
            data = session.query(Poste, Agent) \
                    .join(Poste, Poste.no == Agent.no_poste) \
                    .filter(Poste.no == ut.no_poste_dir).first()
        else:
            data = None

        if data is None:
            poste, agent = Poste(), Agent()
        else:
            poste, agent = data

        result = {}
        for var in ('commune', 'structure', 'poste_resp', 'agent_resp', 'ut', 'agence', 'dt', 'poste', 'agent'):
            result[var] = locals()[var]
        return result


class Poste(DynamicSchemaMixin, Base):
    """Model class for poste table"""
    
    __tablename__ = 'poste'
    __schema__ = PARENT_SCHEMA
    
    no = Column(Unicode(40), primary_key=True)
    intitule = Column(Unicode(80))
    no_service = Column(Unicode(6))
    missions = Column(Unicode(350))
    
    @classmethod
    def get(cls, code, session=PARENT_SESSION):
        if isinstance(code, str):
            code = code.split(',')
        session = get_user_session(session)
        return session.query(cls).filter(cls.no.in_(code))
    
    @classmethod
    def find_by_id(cls, no, session=None):
        session = get_user_session(session) if session is None else session
        params = cls.no.in_(no) if isinstance(no, (list, tuple, set)) else cls.no == no
        return session.query(cls).filter(params).all()


class UtDac(DynamicSchemaMixin, Base):
    """Model class for the territorial units"""
    
    __tablename__ = "ut_dac"
    __schema__ = PARENT_SCHEMA
    
    no_ut = Column(Unicode(40), primary_key=True)
    no_poste = Column(Unicode(40))
    nomat_agent = Column(Unicode(5))
    no_poste_interim = Column(Unicode(40))
    nomat_agent_interim = Column(Unicode(5))
    interim = Column(Unicode(2))
    no_poste_dir = Column(Unicode(40))

    @classmethod
    def get(cls, code, session=PARENT_SESSION):
        if isinstance(code, str):
            code = code.split(',')
        session = get_user_session(session)
        return session.query(cls).filter(cls.no_ut.in_(code))
    
    @classmethod
    def find_by_id(cls, no_ut, session=PARENT_SESSION):
        session = get_user_session(session)
        result = session.query(cls).filter(cls.no_ut == no_ut).first()
        return result if result and result.no_ut else None


class CommuneDac(DynamicSchemaMixin, Base):
    """Model class for communes"""
    
    __tablename__ = 'commune_dac'
    __schema__ = PARENT_SCHEMA
    
    code_commune = Column(Unicode(5), primary_key=True)
    no_ut = Column(Unicode(40))
    no_poste = Column(Unicode(40))
    nomat_agent = Column(Unicode(5))
    interim = Column(Unicode(2))
    
    @classmethod
    def get(cls, code, session=PARENT_SESSION):
        if isinstance(code, str):
            code = code.split(',')
        session = get_user_session(session)
        return session.query(cls).filter(cls.code_commune.in_(code))
    
    @classmethod
    def find_by_id(cls, code_commune, session=PARENT_SESSION):
        session = get_user_session(session)
        result = session.query(cls).filter(cls.code_commune == code_commune).first()
        if result and (getattr(result, 'nomat_agent', None) or getattr(result, 'no_poste', None)):
            return result
        return UtDac.find_by_id(result.no_ut) if result and getattr(result, 'no_ut', None) else None
