#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL). A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS Directory API Module"""

from cornice import Service
from cornice.validators import colander_validator
from pyramid.httpexceptions import HTTPBadRequest, HTTPNotFound, HTTPOk

from onf_website.reference.annuaire.api.interfaces import REST_DIRECTORY_ROUTE
from onf_website.reference.annuaire.api.schema import DirectoryGetterRequest, DirectoryGetterResponse
from onf_website.reference.annuaire.model import Agent
from pyams_security.rest import check_cors_origin, set_cors_headers
from pyams_utils.list import paginate, unique_iter
from pyams_utils.rest import STATUS, http_error, rest_responses

__docformat__ = 'restructuredtext'


def format_contact(agent, poste):
    """Contact card formatter"""
    return {
        "agent_id": agent.nomat,
        "agent_name": f"{agent.prenom} {agent.nom}",
        "intitule_poste": poste.intitule,
        "agent_phones": [
            ' '.join(paginate(phone_number, 2))
            for phone_number in unique_iter(filter(bool, (agent.telephone, agent.portable)))
        ] or None,
        "agent_email": agent.mail or None,
    }


#
# API Directory
#

directory_service = Service(name=REST_DIRECTORY_ROUTE,
                            pyramid_route=REST_DIRECTORY_ROUTE,
                            description='ONF website API for the Directory')


@directory_service.options(validators=(check_cors_origin, set_cors_headers))
def directory_options(request):
    return ''


directory_responses = rest_responses.copy()
directory_responses[HTTPOk.code] = DirectoryGetterResponse(description="Directory search results")


@directory_service.get(schema=DirectoryGetterRequest(),
                       validators=(check_cors_origin, colander_validator, set_cors_headers),
                       response_schemas=directory_responses)
def directory_getter(request):
    insee_code = request.matchdict.get('insee_code')
    if not insee_code:
        return http_error(request, HTTPBadRequest, message="INSEE code is required!")
    contacts = Agent.find_by_insee_code(insee_code)
    if not contacts:
        return http_error(request, HTTPNotFound, message="Contacts not found")
    result = {
        'status': STATUS.SUCCESS.value
    }
    if contacts['agent'].nomat:
        result['contact'] = format_contact(contacts['agent'], contacts['poste'])
    if contacts['agent_resp'].nomat:
        result['manager'] = format_contact(contacts['agent_resp'], contacts['poste_resp'])
    return result

