#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from importlib import import_module

from hypatia.text.lexicon import Lexicon

from onf_website.component.hearing.interfaces import IHearingInfo
from onf_website.component.location.interfaces import ILocationInfo
from onf_website.shared.furniture.interfaces import IFurnitureManagerFactory
from onf_website.shared.planning.interfaces import IWfPlanningInfo
from onf_website.reference.forest import ForestTable
from onf_website.reference.planning import PlanningTable
from onf_website.reference.universe import UniverseTable
from onf_website.reference.location.agency import AgencyTable
from onf_website.reference.location.country import CountryTable
from onf_website.reference.location.department import DepartmentTable
from onf_website.reference.location.region import RegionTable
from onf_website.reference.location.territory import TerritoryTable
from onf_website.reference.location.unit import UnitTable
from onf_website.reference.target import TargetTable
from onf_website.root.interfaces import ISiteRootToolsConfiguration
from onf_website.shared.activity.interfaces import IActivityManagerFactory
from onf_website.shared.casestudy.interfaces import ICaseStudyManagerFactory
from onf_website.shared.contact.interfaces import IContactManagerFactory
from onf_website.shared.expengine.interfaces import IExpEngineManagerFactory
from onf_website.shared.forest.interfaces import IForestManagerFactory
from onf_website.shared.planning.interfaces import IPlanningManagerFactory
from onf_website.shared.hunting.interfaces import IHuntingManagerFactory
from onf_website.shared.interview.interfaces import IInterviewManagerFactory
from onf_website.shared.longread.interfaces import ILongReadManagerFactory
from onf_website.shared.press.interfaces import IPressManagerFactory
from onf_website.shared.report.interfaces import IReportManagerFactory
from onf_website.shared.reserve.interfaces import IReserveManagerFactory
from onf_website.shared.service.interfaces import IServiceManagerFactory
from pyams_catalog.index import FieldIndexWithInterface, KeywordIndexWithInterface
from pyams_catalog.nltk import NltkFullTextProcessor
from pyams_catalog.site import check_required_indexes
from pyams_content.generations import check_required_tables, check_required_tools
from pyams_utils.interfaces.site import ISiteGenerations
from pyams_utils.registry import utility_config
from pyams_utils.site import check_required_utilities


def get_fulltext_lexicon():
    return Lexicon(NltkFullTextProcessor(language='french'))


RENAMED_CLASSES = {
    'onf_website.root SiteRoot': 'pyams_content.root SiteRoot',
    'onf_website.root SiteRootConfiguration':
        'pyams_content.root SiteRootConfiguration',
    'onf_website.root SiteRootBackOfficeConfiguration':
        'pyams_content.root SiteRootBackOfficeConfiguration',
    'onf_website.reference.interfaces IReferenceManager':
        'pyams_content.reference.interfaces IReferenceManager',
    'onf_website.reference ReferencesManager':
        'pyams_content.reference ReferencesManager',
    'onf_website.features.header.interfaces IHeaderLinksContainer':
        'pyams_content.features.menu.interfaces IMenuLinksContainer',
    'onf_website.features.header.interfaces IHeaderLink':
        'pyams_content.features.menu.interfaces IMenuLink',
    'onf_website.features.header.interfaces IHeaderInternalLink':
        'pyams_content.features.menu.interfaces IMenuInternalLink',
    'onf_website.features.header.interfaces IHeaderExternalLink':
        'pyams_content.features.menu.interfaces IMenuExternalLink',
    'onf_website.features.header HeaderMenusContainer':
        'pyams_content.component.association.menu MenusContainer',
    'onf_website.features.header HeaderMenu':
        'pyams_content.component.association.menu Menu',
    'onf_website.features.header HeaderInternalLink':
        'pyams_content.component.links InternalLink',
    'onf_website.features.menu.skin ONFNavigationTabsPortletRendererSettings':
        'onf_website.features.menu.skin ONFNavigationPortletRendererSettingsWithSubtitle',
    'onf_website.features.search.skin HeadFiltesSearchResultsPortletRendererSettings':
        'onf_website.features.search.skin HeadFiltersSearchResultsPortletRendererSettings',
    'onf_website.component.illustration IllustrationParagraphRendererSettings':
        'onf_website.component.illustration.skin IllustrationParagraphRendererSettings',
    'onf_website.component.paragraph.interfaces.map IMapParagraph':
        'onf_website.component.map.interfaces IMapParagraph',
    'onf_website.component.paragraph.map MapParagraph':
        'onf_website.component.map.paragraph MapParagraph',
    'onf_website.component.paragraph.skin.interfaces.map IMapParagraphDefaultRendererSettings':
        'onf_website.component.map.skin.interfaces IMapParagraphDefaultRendererSettings',
    'onf_website.component.paragraph.skin.map MapParagraphDefaultRendererSettings':
        'onf_website.component.map.skin MapParagraphDefaultRendererSettings',
    'onf_website.shared.common.specificities SharedContentSpecificitiesParagraph':
        'pyams_content.shared.common.specificities SharedContentSpecificitiesParagraph',
    'onf_website.shared.common.specificities SharedContentSpecificitiesPortletSettings':
        'pyams_content.shared.common.portlet.specificities '
        'SharedContentSpecificitiesPortletSettings',
    'onf_website.shared.site.manager SiteManager':
        'pyams_content.shared.site.manager SiteManager',
    'onf_website.shared.site.folder SiteFolder': 'pyams_content.shared.site.folder SiteFolder',
    'onf_website.shared.site Topic': 'pyams_content.shared.site Topic',
    'onf_website.shared.site WfTopic': 'pyams_content.shared.site WfTopic',
    'onf_website.shared.blog.manager BlogManager':
        'pyams_content.shared.blog.manager BlogManager',
    'onf_website.shared.blog BlogPost': 'pyams_content.shared.blog BlogPost',
    'onf_website.shared.blog WfBlogPost': 'pyams_content.shared.blog WfBlogPost',
    'onf_website.shared.common.skin.head HubHeaderWithImagesPortletRendererSettings':
        'onf_website.shared.hub.skin.head HubHeaderWithImagesPortletRendererSettings',
    'onf_website.shared.news.manager NewsManager':
        'pyams_content.shared.news.manager NewsManager',
    'onf_website.shared.news NewsEvent': 'pyams_content.shared.news NewsEvent',
    'onf_website.shared.news WfNewsEvent': 'pyams_content.shared.news WfNewsEvent',
    'onf_website.shared.resource.interfaces IResourceManager':
        'pyams_content.shared.resource.interfaces IResourceManager',
    'onf_website.shared.resource.manager ResourceManager':
        'pyams_content.shared.resource.manager ResourceManager',
    'onf_website.shared.resource Resource': 'pyams_content.shared.resource Resource',
    'onf_website.shared.resource ResourceInfo': 'pyams_content.shared.resource ResourceInfo',
    'onf_website.shared.resource WfResource': 'pyams_content.shared.resource WfResource',
    'onf_website.shared.usecase.interfaces IUseCaseManager':
        'onf_website.shared.casestudy.interfaces ICaseStudyManager',
    'onf_website.shared.usecase.manager UseCaseManager':
        'onf_website.shared.casestudy.manager CaseStudyManager',
}

REQUIRED_UTILITIES = ()

REQUIRED_TABLES = (
    ('country_table_name', 'countries', CountryTable),
    ('region_table_name', 'regions', RegionTable),
    ('department_table_name', 'departments', DepartmentTable),
    ('territory_table_name', 'territories', TerritoryTable),
    ('agency_table_name', 'agencies', AgencyTable),
    ('unit_table_name', 'units', UnitTable),
    ('target_table_name', 'targets', TargetTable),
    ('forest_table_name', 'forests', ForestTable),
    ('planning_table_name', 'plannings', PlanningTable),
    ('universe_table_name', 'universe', UniverseTable)
)

REQUIRED_TOOLS = (
    ('reports', IReportManagerFactory),
    ('interviews', IInterviewManagerFactory),
    ('expengines', IExpEngineManagerFactory),
    ('longreads', ILongReadManagerFactory),
    ('activities', IActivityManagerFactory),
    ('casestudies', ICaseStudyManagerFactory),
    ('services', IServiceManagerFactory),
    ('press', IPressManagerFactory),
    ('contacts', IContactManagerFactory),
    ('hunting', IHuntingManagerFactory),
    ('forests', IForestManagerFactory),
    ('plannings', IPlanningManagerFactory),
    ('reserves', IReserveManagerFactory),
    ('furniture', IFurnitureManagerFactory)
)

REQUIRED_INDEXES = (
    ('forests', KeywordIndexWithInterface, {'interface': ILocationInfo,
                                            'discriminator': 'forests_index'}),
    ('plannings', KeywordIndexWithInterface, {'interface': IWfPlanningInfo,
                                              'discriminator': 'planning_ids'}),
    ('cities', KeywordIndexWithInterface, {'interface': ILocationInfo,
                                           'discriminator': 'cities'}),
    ('departments', KeywordIndexWithInterface, {'interface': ILocationInfo,
                                                'discriminator': 'departments_index'}),
    ('countries', KeywordIndexWithInterface, {'interface': ILocationInfo,
                                              'discriminator': 'countries_index'}),
    ('structures', KeywordIndexWithInterface, {'interface': ILocationInfo,
                                               'discriminator': 'structures'}),
    ('targets', KeywordIndexWithInterface, {'interface': IHearingInfo,
                                            'discriminator': 'targets_index'}),
    ('national_scope', FieldIndexWithInterface, {'interface': IHearingInfo,
                                                 'discriminator': 'national_scope'}),
    ('hearing_forests', KeywordIndexWithInterface, {'interface': IHearingInfo,
                                                    'discriminator': 'forests_index'}),
    ('hearing_cities', KeywordIndexWithInterface, {'interface': IHearingInfo,
                                                   'discriminator': 'cities'}),
    ('hearing_departments', KeywordIndexWithInterface, {'interface': IHearingInfo,
                                                        'discriminator': 'departments_index'}),
    ('hearing_countries', KeywordIndexWithInterface, {'interface': IHearingInfo,
                                                      'discriminator': 'countries_index'}),
    ('hearing_structures', KeywordIndexWithInterface, {'interface': IHearingInfo,
                                                       'discriminator': 'structures_index'}),
    ('source_site', FieldIndexWithInterface, {'interface': IHearingInfo,
                                              'discriminator': 'source_index'}),
    ('diffusion_sites', KeywordIndexWithInterface, {'interface': IHearingInfo,
                                                    'discriminator': 'diffusion_sites_index'})
)


@utility_config(name='ONF website', provides=ISiteGenerations)
class WebsiteGenerationsChecker(object):
    """ONF website generations checker"""

    order = 200
    generation = 6

    def evolve(self, site, current=None):
        """Check for required utilities"""
        check_required_utilities(site, REQUIRED_UTILITIES)
        check_required_indexes(site, REQUIRED_INDEXES)
        check_required_tables(site, REQUIRED_TABLES)
        check_required_tools(site, ISiteRootToolsConfiguration, REQUIRED_TOOLS)

        if not current:
            current = 1
        for generation in range(current, self.generation):
            module_name = 'onf_website.generations.evolve{}'.format(generation)
            module = import_module(module_name)
            module.evolve(site)
