#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import requests
from persistent import Persistent
from zope.schema.fieldproperty import FieldProperty

from onf_website.features.zfiles.interfaces import IZFilesClientInfo, IZFilesClientTarget
from pyams_file.file import TempFile
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config


@factory_config(IZFilesClientInfo)
class ZFilesClientInfo(Persistent):
    """ZFiles client info persistent class"""

    zfiles_url = FieldProperty(IZFilesClientInfo['zfiles_url'])
    login = FieldProperty(IZFilesClientInfo['login'])
    password = FieldProperty(IZFilesClientInfo['password'])
    verify = FieldProperty(IZFilesClientInfo['verify'])

    def get_target_url(self):
        """Get base ZFiles target URL"""
        url = self.zfiles_url
        return f"{url}{'' if url.endswith('/') else '/'}api/zfiles/rest"

    def get_existing_docs(self, oids):
        """Get list of existing docs from given OIDs"""
        if not (self.zfiles_url and oids):
            return []
        try:
            oids_params = '&'.join((
                f'oid={oid}'
                for oid in oids
            )) + '&oid='
            res = requests.get(f'{self.get_target_url()}?{oids_params}'
                               f'&fields=oid,filesize,filename',
                               auth=(self.login, self.password),
                               verify=self.verify)
            if res.status_code != 200:
                return []
            return res.json().get('results', [])
        except (requests.ConnectionError, requests.Timeout):
            return []

    def get_document(self, oid):
        """Get document content for given OID"""
        if not (self.zfiles_url and oid):
            return None
        try:
            props_req = requests.get(f'{self.get_target_url()}/{oid}'
                                     f'?fields=filename,content_type,href,updated_time',
                                     auth=(self.login, self.password),
                                     verify=self.verify)
            if props_req.status_code != 200:
                return None
            props = props_req.json()
            if not props.get('href'):
                return None
    
            doc_req = requests.get(props['href'],
                                   auth=(self.login, self.password),
                                   verify=self.verify)
            if doc_req.status_code != 200:
                return None
            return TempFile(data=doc_req.content,
                            content_type=props.get('content_type'),
                            filename=props.get('filename'),
                            modification_date=props.get('updated_time'))
        except (requests.ConnectionError, requests.Timeout):
            return None


ZFILES_CLIENT_INFO_KEY = 'onf_website.zfiles'


@adapter_config(context=IZFilesClientTarget,
                provides=IZFilesClientInfo)
def zfiles_client_factory(context):
    """ZFiles client factory"""
    return get_annotation_adapter(context, ZFILES_CLIENT_INFO_KEY, IZFilesClientInfo)
