#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""ONF_Website.features.tarteaucitron module

This module defines TarteAuCitron integration components, which is used to handle
cookies information banners.
"""

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.container.contained import Contained
from zope.schema.fieldproperty import FieldProperty

from onf_website.features.tarteaucitron.interfaces import ITarteAuCitronInfo, \
    TARTEAUCITRON_ANNOTATION_KEY
from pyams_content.root import ISiteRoot
from pyams_sequence.reference import get_reference_target
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.url import canonical_url


def asbool(value):
    return 'true' if value else 'false'


@factory_config(ITarteAuCitronInfo)
class TarteAuCitronInfo(Persistent, Contained):
    """TarteAuCitron configuration class"""

    enabled = FieldProperty(ITarteAuCitronInfo['enabled'])
    policy_target = FieldProperty(ITarteAuCitronInfo['policy_target'])
    hashtag = FieldProperty(ITarteAuCitronInfo['hashtag'])
    cookie_nam = FieldProperty(ITarteAuCitronInfo['cookie_name'])
    orientation = FieldProperty(ITarteAuCitronInfo['orientation'])
    group_services = FieldProperty(ITarteAuCitronInfo['group_services'])
    show_alert_small = FieldProperty(ITarteAuCitronInfo['show_alert_small'])
    cookies_list = FieldProperty(ITarteAuCitronInfo['cookies_list'])
    close_popup = FieldProperty(ITarteAuCitronInfo['close_popup'])
    show_icon = FieldProperty(ITarteAuCitronInfo['show_icon'])
    icon_position = FieldProperty(ITarteAuCitronInfo['icon_position'])
    adblocker = FieldProperty(ITarteAuCitronInfo['adblocker'])
    deny_all = FieldProperty(ITarteAuCitronInfo['deny_all'])
    accept_all = FieldProperty(ITarteAuCitronInfo['accept_all'])
    mandatory = FieldProperty(ITarteAuCitronInfo['mandatory'])
    high_privacy = FieldProperty(ITarteAuCitronInfo['high_privacy'])
    remove_credit = FieldProperty(ITarteAuCitronInfo['remove_credit'])
    more_info = FieldProperty(ITarteAuCitronInfo['more_info'])
    more_info_link = FieldProperty(ITarteAuCitronInfo['more_info_link'])

    services = FieldProperty(ITarteAuCitronInfo['services'])

    def get_policy_target(self, request):
        target = get_reference_target(self.policy_target)
        if target is not None:
            return canonical_url(target, request)
        return ''

    def get_info_target(self, request):
        target = get_reference_target(self.more_info_link)
        if target is not None:
            return canonical_url(target, request)
        return ''

    def to_json(self, request):
        return """{{
            "privacyUrl": "{1}",
            "hashtag": "{0.hashtag}",
            "cookieName": "{0.cookie_name}",
            "orientation": "{0.orientation}",
            "groupServices": {2},
            "showAlertSmall": {3},
            "cookiesList": {4},
            "closePopup": {5},
            "showIcon": {6},
            "iconPosition": "{0.icon_position}",
            "adblocker": {7},
            "DenyAllCta": {8},
            "AcceptAllCta": {9},
            "mandatory": {10},
            "highPrivacy": {11},
            "removeCredit": {12},
            "moreInfoLink": {13},
            "readmoreLink": "{14}",
            "useExternalCss": false,
            "useExternalJs": false
        }}""".format(self,
                     self.get_policy_target(request),
                     asbool(self.group_services),
                     asbool(self.show_alert_small),
                     asbool(self.cookies_list),
                     asbool(self.close_popup),
                     asbool(self.show_icon),
                     asbool(self.adblocker),
                     asbool(self.deny_all),
                     asbool(self.accept_all),
                     asbool(self.mandatory),
                     asbool(self.high_privacy),
                     asbool(self.remove_credit),
                     asbool(self.more_info),
                     self.get_info_target(request))


@adapter_config(context=ISiteRoot, provides=ITarteAuCitronInfo)
def site_root_tarteaucitron_adapter(context):
    """Site root TarteAuCitron adapter"""
    return get_annotation_adapter(context, TARTEAUCITRON_ANNOTATION_KEY,
                                  ITarteAuCitronInfo)
