#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import json

from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from z3c.form.field import Fields
from zope.interface import Interface, implementer
from zope.intid.interfaces import IIntIds
from zope.location.interfaces import ISublocations
from zope.schema import getFieldsInOrder
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from onf_website.component.location.skin.map import BaseMapMarkersRendererMixin
from onf_website.features.search.skin import AggregatedSearchResultsRenderer
from onf_website.features.search.skin.interfaces import IMapWithHeadFiltersSearchResultsPortletRendererSettings
from onf_website.features.search.skin.interfaces.filter import IFilterAggregate, IFilterProcessor
from onf_website.features.search.skin.interfaces.nearme import INearmeFilterContainer, INearmeFilterContainerTarget, \
    INearmeMapSearchResultsPortletRendererSettings, INearmeSearchResultsPortletRendererSettings, \
    NEARME_FILTER_CONTAINER_ANNOTATION_KEY
from onf_website.reference.forest.model.foret import Foret, REG_ID_NAT_FRT
from onf_website.reference.insee.model import Commune, REG_COMMUNE_CODE
from onf_website.reference.universe.interfaces import IUniverseTable
from onf_website.skin.public import aside_sticky
from pyams_content.features.search.portlet.interfaces import IAggregatedPortletRenderer, ISearchResultsPortletSettings
from pyams_content.shared.common import IWfSharedContent
from pyams_default_theme.features.search.portlet import SearchResultsPortletRendererBaseSettings
from pyams_default_theme.features.search.portlet.interfaces import ISearchResultRenderer
from pyams_default_theme.features.search.portlet.interfaces import ISearchResultTarget, ISearchResultTitle
from pyams_default_theme.interfaces import ISearchResultsView
from pyams_file.property import FileProperty
from pyams_form.group import NamedWidgetsGroup
from pyams_form.interfaces.form import IFormManager
from pyams_form.manager import DialogEditFormFormManager
from pyams_gis.configuration import MapConfiguration
from pyams_gis.interfaces.configuration import IMapConfiguration
from pyams_gis.interfaces.utility import IMapManager
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_portal.zmi.portlet import PortletRendererPropertiesEditForm
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer, IPyAMSUserLayer
from pyams_template.template import template_config
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.container import BTreeOrderedContainer
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.url import canonical_url
from pyams_viewlet.viewlet import ViewContentProvider

from onf_website import _


#
# Near-me filters container
#

@factory_config(INearmeFilterContainer)
class NearmeFilterContainer(BTreeOrderedContainer):
    """Near-me filter container"""

    def add(self, obj):
        intids = get_utility(IIntIds)
        self[hex(intids.register(obj))[2:]] = obj

    def get_visible_filters(self):
        yield from filter(lambda x: x.visible, self.values())


@adapter_config(required=INearmeFilterContainerTarget,
                provides=INearmeFilterContainer)
def nearme_filter_container(context):
    """Near-me filter container adapter"""
    return get_annotation_adapter(context, NEARME_FILTER_CONTAINER_ANNOTATION_KEY,
                                  INearmeFilterContainer, name='++filter.nearme++')


@adapter_config(name='filter.nearme',
                required=INearmeFilterContainerTarget,
                provides=ITraversable)
class NearmeFilterContainerTraverser(ContextAdapter):
    """Near-me filter container traverser"""

    def traverse(self, name, furtherPath=None):
        container = INearmeFilterContainer(self.context, None)
        if name:
            return container.get(name)
        return container


@adapter_config(name='filter.nearme',
                required=INearmeFilterContainerTarget,
                provides=ISublocations)
class NearmeFilterContainerSublocations(ContextAdapter):
    """Near-me filter container sublocations"""

    def sublocations(self):
        return INearmeFilterContainer(self.context).values()



class UniverseTableMeta(type):
    def __new__(cls, name, bases, dct):
        table = query_utility(IUniverseTable)
        if table:
            for attr_name in dir(table):
                if not attr_name.startswith('_'):
                    dct[attr_name] = property(cls._create_property(attr_name))
        return super(UniverseTableMeta, cls).__new__(cls, name, bases, dct)

    @staticmethod
    def _create_property(attr_name):
        def prop(self):
            table = query_utility(IUniverseTable)
            if table is None:
                raise AttributeError(f"IUniverseTable utility not found.")
            if not hasattr(table, attr_name):
                raise AttributeError(f"'{type(table).__name__}' object has no attribute '{attr_name}'")
            return getattr(table, attr_name)
        return prop
#
# Advanced filters portlet renderer settings
#

@factory_config(INearmeSearchResultsPortletRendererSettings)
@implementer(INearmeFilterContainerTarget, IAggregatedPortletRenderer)
class NearmeSearchResultsPortletRendererSettings(SearchResultsPortletRendererBaseSettings):
    """Search results portlet advanced filters renderer settings"""
    
    presentation_mode_reference = FieldProperty(INearmeSearchResultsPortletRendererSettings['presentation_mode_reference'])
    description = FieldProperty(INearmeSearchResultsPortletRendererSettings['description'])
    dropdown_menu_title = FieldProperty(INearmeSearchResultsPortletRendererSettings['dropdown_menu_title'])
    region_title =  FieldProperty(INearmeSearchResultsPortletRendererSettings['region_title'])

    @property
    def filters(self):
        yield from INearmeFilterContainer(self, {}).get_visible_filters()
    
    def get_labels_for_tag_ids(self, tag_data):
        intids = get_utility(IIntIds)
        result = {}
        for item in tag_data:
            try:
                tag_id = int(item['key'])
                tag_object = intids.queryObject(tag_id)
                if tag_object is not None:
                    result[tag_id] = tag_object.label
            except (KeyError, ValueError, TypeError):
                continue
        return result

    @property
    def universe_table_config(self):
         table = query_utility(IUniverseTable)
         return {
             "dropdown_menu_title": II18n(table).query_attribute('dropdown_menu_title'),
             "inactive_info_title": II18n(table).query_attribute('inactive_info_title'),
             "inactive_info_text": II18n(table).query_attribute('inactive_info_text'),
             "online_help_title": II18n(table).query_attribute('online_help_title'),
             "online_help_text": II18n(table).query_attribute('online_help_text'),
         }

    @property
    def universes_list(self):
        table = query_utility(IUniverseTable)
        if table is not None:
            request = check_request()
            terms = [{
                'title': II18n(v).query_attribute('title', request=request),
                'url': canonical_url(get_reference_target(v.contact_target), request)
            } for v in table.values()]
        else:
            terms = []
        return terms

    
    def processed_filters(self, request, aggregations):
        processed = []
        for filter in self.filters:
            processor = request.registry.queryAdapter(filter, IFilterProcessor)
            if processor is not None:
                processed_data = processor.process(request, aggregations)
                processed.append(processed_data)
        return processed

    @property
    def aggregates(self):
        return [
            IFilterAggregate(filter)
            for filter in self.filters
        ]


@adapter_config(name='nearme-list-results',
                required=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/nearme-results.pt',
                 layer=IPyAMSLayer)
@implementer(ISearchResultsView)
class NearmeSearchResultsPortletResultsRenderer(AggregatedSearchResultsRenderer):
    """Advanced filters search results portlet renderer"""

    label = _("ONF: list of results 'Near me'")
    weight = 65

    settings_interface = INearmeSearchResultsPortletRendererSettings

    @property
    def presentation_mode_reference(self):
        return get_reference_target(self.renderer_settings.presentation_mode_reference, request=self.request)
    
    def update(self):
        super().update()
        aside_sticky.need()
    
    def clean_url(slef , url):
        url = url.split('?')[0]
        url = url.split('#')[0]
        if url.endswith('/preview.html'):
            url = url[:-13]
        return url


@adapter_config(context=(INearmeMapSearchResultsPortletRendererSettings, IPyAMSLayer,
                         PortletRendererPropertiesEditForm),
                provides=IFormManager)
class INearmeMapSearchResultsPortletRendererSettingsFormManager(DialogEditFormFormManager):
    """Near-me search results portlet map renderer settings properties edit form manager"""
    presentation_mode_reference = FieldProperty(
            INearmeMapSearchResultsPortletRendererSettings['presentation_mode_reference'])

    def getFields(self):
        fields = Fields(INearmeMapSearchResultsPortletRendererSettings).omit(
            'use_default_map_configuration')
        fields['no_use_default_map_configuration'].widgetFactory = SingleCheckBoxFieldWidget
        return fields

    def updateGroups(self):
        form = self.view
        form.add_group(NamedWidgetsGroup(form, 'configuration', form.widgets,
                                         ('no_use_default_map_configuration', ) + tuple([
                                             name
                                             for name, field in getFieldsInOrder(IMapConfiguration)
                                         ]),
                                         legend=_("Don't use default map configuration"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=INearmeMapSearchResultsPortletRendererSettings[
                                             'no_use_default_map_configuration']))
        super().updateGroups()


@factory_config(INearmeMapSearchResultsPortletRendererSettings)
class NearmeMapSearchResultsPortletRendererSettings(NearmeSearchResultsPortletRendererSettings, MapConfiguration):
    """Map search results portlet renderer settings"""
    
    map_height = FieldProperty(
            IMapWithHeadFiltersSearchResultsPortletRendererSettings['map_height'])
    display_illustrations = FieldProperty(
            IMapWithHeadFiltersSearchResultsPortletRendererSettings['display_illustrations'])
    marker_image = FileProperty(
            IMapWithHeadFiltersSearchResultsPortletRendererSettings['marker_image'])
    display_clusters = FieldProperty(
            IMapWithHeadFiltersSearchResultsPortletRendererSettings['display_clusters'])
    
    presentation_mode_reference = FieldProperty(
            INearmeMapSearchResultsPortletRendererSettings['presentation_mode_reference'])

    _use_default_map_configuration = FieldProperty(
            INearmeMapSearchResultsPortletRendererSettings['use_default_map_configuration'])
    
    allow_pagination = False

    @property
    def use_default_map_configuration(self):
        return self._use_default_map_configuration

    @use_default_map_configuration.setter
    def use_default_map_configuration(self, value):
        self._use_default_map_configuration = value

    @property
    def no_use_default_map_configuration(self):
        return not bool(self.use_default_map_configuration)

    @no_use_default_map_configuration.setter
    def no_use_default_map_configuration(self, value):
        self.use_default_map_configuration = not bool(value)

    @property
    def configuration(self):
        if self.use_default_map_configuration:
            manager = get_utility(IMapManager)
            return IMapConfiguration(manager).get_configuration()
        return self.get_configuration()


@adapter_config(name='nearmy-map-filters',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/nearme-search-map.pt', layer=IPyAMSLayer)
class NearmyMapSearchResultsPortletRenderer(BaseMapMarkersRendererMixin,
                                            AggregatedSearchResultsRenderer):
    """Search results portlet map with head filters renderer"""

    label = _("ONF: search map 'Near me'")

    settings_interface = INearmeMapSearchResultsPortletRendererSettings
    weight = 70

    default_page_length = 999

    @property
    def presentation_mode_reference(self):
        return get_reference_target(self.renderer_settings.presentation_mode_reference, request=self.request)
    
    @property
    def values_map(self):
        user_search = self.request.params.get('user_search', '').strip()
        if user_search:
            if REG_ID_NAT_FRT.match(user_search.upper()):
                foret, info_foret = Foret.find_by_id(user_search.upper(), with_info=True).first()
                if foret is not None:
                    return json.dumps({
                        user_search: getattr(info_foret, 'libelle', None) or foret.title
                    })
            elif REG_COMMUNE_CODE.match(user_search):
                commune = Commune.get(user_search).first()
                if commune is not None:
                    return json.dumps({user_search: commune.title})
        return json.dumps({})

    def clean_url(slef , url):
        url = url.split('?')[0]
        url = url.split('#')[0]
        if url.endswith('/preview'):
            url = url[:-8]
        return url

    def get_markers(self, results, marker_image=None):
        return super().get_markers(results, self.renderer_settings.marker_image)


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer, NearmeSearchResultsPortletResultsRenderer),
                provides=ISearchResultRenderer)
@template_config(template='templates/nearme-card.pt', layer=IPyAMSUserLayer)
class WFNearmeSearchResultsPortletListOfResultsRenderer(ViewContentProvider):
    """Shared content search result renderer"""

    @property
    def url(self):
        return self.request.registry.queryMultiAdapter((self.context, self.request, self.view),
                                                       ISearchResultTarget)

    @property
    def title(self):
        return self.request.registry.queryMultiAdapter((self.context, self.request, self.view),
                                                       ISearchResultTitle)
