#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from collections import OrderedDict

from zope.annotation.interfaces import IAttributeAnnotatable
from zope.container.constraints import contains
from zope.container.interfaces import IContainer
from zope.interface import Attribute, Interface
from zope.schema import Bool, Choice, Int, List
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_default_theme.features.search.portlet import ISearchResultsPortletRendererBaseSettings
from pyams_i18n.schema import I18nTextLineField

from onf_website import _


FILTER_CONTAINER_ANNOTATION_KEY = 'onf_website.search.filters'


FILTER_SORTING = OrderedDict((
    ('alphabetical_asc', _("Alphabetical")),
    ('alphabetical_desc', _("Alphabetical (reversed)")),
    ('count_asc', _("Results count")),
    ('count_desc', _("Results count (reversed)"))
))

FILTER_SORTING_VOCABULARY = SimpleVocabulary([
    SimpleTerm(k, title=v)
    for k, v in FILTER_SORTING.items()
])


MANUAL_FILTER_SORTING = FILTER_SORTING.copy()
MANUAL_FILTER_SORTING['manual'] = _("Manual")

MANUAL_FILTER_SORTING_VOCABULARY = SimpleVocabulary([
    SimpleTerm(k, title=v)
    for k, v in MANUAL_FILTER_SORTING.items()
])


DISPLAY_MODE = (
    ('list', _("List")),
    ('select', _("Choice")),
)

DISPLAY_MODE_VOCABULARY = SimpleVocabulary([
    SimpleTerm(key, title=value)
    for key, value in DISPLAY_MODE
])


CONTENT_TYPE_OPTIONS = {
    'facet_label': _("Content-type label"),
    'facet_type_label': _("Data type label")
}

CONTENT_TYPE_VOCABULARY = SimpleVocabulary([
    SimpleTerm(key, title=value)
    for key, value in CONTENT_TYPE_OPTIONS.items()
])


class IFilter(Interface):
    """Base filter interface"""

    visible = Bool(title=_("Visible?"),
                   description=_("Is this item visible in front-office?"),
                   required=True,
                   default=True)

    label = I18nTextLineField(title=_("Label"),
                              description=_("This is the label displayed in filter header"),
                              required=True)
    
    display_mode = Choice(title=_("Display mode"),
                          description=_("Choose the display mode for entries"),
                          vocabulary=DISPLAY_MODE_VOCABULARY,
                          default='list',
                          required=True)
    
    displayed_entries = Int(title=_("Displayed entries"),
                            description=_("Number of entries displayed in search filter"),
                            required=True,
                            default=5)
    
    select_placeholder = I18nTextLineField(title=_("Placeholder for select"),
                                           description=_("Placeholder text to be displayed in the select widget"),
                                           required=False)
    
    sorting_options = Choice(title=_("Sorting mode"),
                             description=_("Filter entries sorting mode"),
                             vocabulary=FILTER_SORTING_VOCABULARY,
                             default='alphabetical_asc',
                             required=True)
    
    filter_type = Attribute("Filter type")
    
    def is_visible(self, request=None):
        """Is association item published?"""


class ITagFilter(IFilter):
    """Tag Filter Interface"""

    thesaurus_name = Choice(title=_("Thesaurus name"),
                            vocabulary='PyAMS thesaurus names',
                            required=True)

    extract_name = Choice(title=_("Thesaurus extract"),
                          vocabulary='PyAMS thesaurus extracts',
                          required=False)


class ICollectionFilter(IFilter):
    """Collection Filter Interface"""

    thesaurus_name = Choice(title=_("Thesaurus name"),
                            vocabulary='PyAMS thesaurus names',
                            required=True)

    extract_name = Choice(title=_("Thesaurus extract"),
                          vocabulary='PyAMS thesaurus extracts',
                          required=False)


class IThemeFilter(IFilter):
    """Theme Filter Interface"""

    thesaurus_name = Choice(title=_("Thesaurus name"),
                            vocabulary='PyAMS thesaurus names',
                            required=True)

    extract_name = Choice(title=_("Thesaurus extract"),
                          vocabulary='PyAMS thesaurus extracts',
                          required=False)


class ITargetFilter(IFilter):
    """Target filter Interface"""

    targets = List(title=_("Other targets"),
                   value_type=Choice(vocabulary='ONF targets'),
                   required=True)

    sorting_options = Choice(title=_("Sorting mode"),
                             description=_("Filter entries sorting mode"),
                             vocabulary=MANUAL_FILTER_SORTING_VOCABULARY,
                             default='alphabetical_asc',
                             required=True)


class ILocationFilter(IFilter):
    """location filter Interface"""


class IContentTypeFilter(IFilter):
    """Content-type filter interface"""

    content_type_display_mode = Choice(title=_("Content type display mode"),
                                       description=_("Choose the display mode for content "
                                                     "type entries"),
                                       vocabulary=CONTENT_TYPE_VOCABULARY,
                                       default='facet_label',
                                       required=True)


class IFilterContainer(IContainer):
    """Filter container interface"""

    contains(IFilter)

    def add(self, obj):
        """Add filter to container"""

    def get_visible_filters(self):
        """Visible filters iterator"""


class IFilterContainerTarget(IAttributeAnnotatable):
    """Filter container target marker interface"""


class ISearchResultsPortletAdvancedFiltersRendererSettings(ISearchResultsPortletRendererBaseSettings):
    """Advanced filters search results portlet renderer settings"""

    filters = Attribute("Search filters")


class IFilterAggregate(Interface):
    """Filter aggregate getter"""


class IFilterType(Interface):
    """Filter type interface"""


class IFilterProcessor(Interface):
    """Filter processor interface"""

    def process(self, request):
        """ Process the filter and return relevant data"""
