#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import locale

from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import Interface, implementer
from zope.intid.interfaces import IIntIds
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from onf_website.features.search.skin import AggregatedSearchResultsRenderer
from onf_website.features.search.skin.interfaces.filter import FILTER_CONTAINER_ANNOTATION_KEY, ICollectionFilter, \
    IContentTypeFilter, IFilter, IFilterAggregate, IFilterContainer, IFilterContainerTarget, IFilterProcessor, \
    ILocationFilter, ISearchResultsPortletAdvancedFiltersRendererSettings, ITagFilter, ITargetFilter, IThemeFilter
from onf_website.features.thesaurus import IThesaurusTermFilterInfo
from onf_website.reference.location.interfaces.department import IDepartmentTable
from onf_website.skin.public import aside_sticky
from pyams_content.features.search import ISearchFolder, ISearchFormRequestParams
from pyams_content.features.search.portlet.interfaces import IAggregatedPortletRenderer, ISearchResultsPortletSettings
from pyams_default_theme.features.search.portlet import SearchResultsPortletRendererBaseSettings
from pyams_default_theme.interfaces import ISearchResultsView
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer, MANAGE_TEMPLATE_PERMISSION
from pyams_skin.layer import IPyAMSLayer, IPyAMSUserLayer
from pyams_template.template import template_config
from pyams_thesaurus.interfaces.thesaurus import IThesaurus, IThesaurusExtracts
from pyams_utils.adapter import ContextAdapter, ContextRequestAdapter, adapter_config, get_annotation_adapter
from pyams_utils.container import BTreeOrderedContainer
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import query_request

from onf_website import _


def get_sorting_params(sorting_option):
    """Returns the sorting parameters for Elasticsearch queries based on the provided sorting option.

    :param sorting_option: A string representing the desired sorting option.
    :return: A dictionary representing the Elasticsearch sorting parameters.
    """
    if sorting_option == 'alphabetical_desc':
        sorting_params = {'_key': 'desc'}
    elif sorting_option == 'count_desc':
        sorting_params = {'_count': 'desc'}
    elif sorting_option == 'count_asc':
        sorting_params = {'_count': 'asc'}
    else:
        sorting_params = {'_key': 'asc'}
    return sorting_params


@implementer(IFilter)
class Filter(Persistent, Contained):
    """Base filter persistent class"""

    visible = FieldProperty(IFilter['visible'])
    label = FieldProperty(IFilter['label'])
    display_mode = FieldProperty(IFilter['display_mode'])
    displayed_entries = FieldProperty(IFilter['displayed_entries'])
    select_placeholder = FieldProperty(IFilter['select_placeholder'])
    sorting_options = FieldProperty(IFilter['sorting_options'])


#
# Content-type filter
#

@factory_config(IContentTypeFilter)
class ContentTypeFilter(Filter):
    """Content-type filter """

    content_type_display_mode = FieldProperty(IContentTypeFilter['content_type_display_mode'])

    filter_type = 'facet_label'


@adapter_config(required=IContentTypeFilter,
                provides=IFilterAggregate)
def content_type_filter_aggregate(context):
    """Content-type filter aggregate getter"""
    sorting_option = context.sorting_options
    sorting_params = get_sorting_params(sorting_option)
    display_mode = context.content_type_display_mode
    return {
        'name': display_mode,
        'type': 'terms',
        'params': {
            'field': display_mode,
            'size': 100,
            'order': sorting_params
        }
    }


def get_filter_aggregate(context, filter_name, field_name):
    """Utility function to build thesaurus filter aggregate."""
    sorting_option = context.sorting_options
    sorting_params = get_sorting_params(sorting_option)
    thesaurus = query_utility(IThesaurus, name=context.thesaurus_name)
    filter_aggregate = {
        'name': filter_name,
        'type': 'terms',
        'params': {
            'field': field_name,
            'size': 100,
            'order': sorting_params
        }
    }
    if context.extract_name:
        extract = IThesaurusExtracts(thesaurus).get(context.extract_name)
        terms = list(extract.terms_ids)
        if terms:
            filter_aggregate['params']['include'] = terms
    return filter_aggregate


@adapter_config(name='facet_label',
                context=(ISearchFolder, IPyAMSUserLayer),
                provides=ISearchFormRequestParams)
class SearchFolderFacetLabelRequestParams(ContextRequestAdapter):
    """Search folder facet label request params"""
    
    def get_params(self):
        for label in self.request.params.getall('facet_label'):
            yield {
                'name': 'facet_label',
                'value': label
            }


@adapter_config(name='facet_type_label',
                context=(ISearchFolder, IPyAMSUserLayer),
                provides=ISearchFormRequestParams)
class EsSearchFolderFacetTypeLabelRequestParams(ContextRequestAdapter):
    """Search folder facet type label request params"""
    
    def get_params(self):
        for label in self.request.params.getall('facet_type_label'):
            yield {
                'name': 'facet_type_label',
                'value': label
            }


@factory_config(ITagFilter)
class TagsFilter(Filter):
    """tags filter"""

    thesaurus_name = FieldProperty(ITagFilter['thesaurus_name'])
    extract_name = FieldProperty(ITagFilter['extract_name'])

    filter_type = 'tag'


@adapter_config(required=ITagFilter,
                provides=IFilterAggregate)
def tags_filter_aggregate(context):
    """tags filter aggregate getter"""
    return get_filter_aggregate(context, 'tag', 'tags')


@factory_config(ICollectionFilter)
class CollectionFilter(Filter):
    """Collection filter"""

    thesaurus_name = FieldProperty(ICollectionFilter['thesaurus_name'])
    extract_name = FieldProperty(ICollectionFilter['extract_name'])

    filter_type = 'collection'


@adapter_config(required=ICollectionFilter,
                provides=IFilterAggregate)
def collection_filter_aggregate(context):
    """Content-type filter aggregate getter"""
    return get_filter_aggregate(context, 'collection', 'collections')


@factory_config(IThemeFilter)
class ThemeFilter(Filter):
    """theme filter"""

    thesaurus_name = FieldProperty(IThemeFilter['thesaurus_name'])
    extract_name = FieldProperty(IThemeFilter['extract_name'])

    filter_type = 'theme'


@adapter_config(required=IThemeFilter,
                provides=IFilterAggregate)
def theme_filter_aggregate(context):
    """theme filter aggregate getter"""
    return get_filter_aggregate(context, 'theme', 'themes.terms')


@factory_config(ITargetFilter)
class TargetFilter(Filter):
    """Target filter"""

    targets = FieldProperty(ITargetFilter['targets'])

    filter_type = 'hearing'
    

@adapter_config(required=ITargetFilter,
                provides=IFilterAggregate)
def target_filter_aggregate(context):
    """Target filter aggregate getter"""
    sorting_option = context.sorting_options
    # Assuming get_sorting_params returns a dict suitable for ES sorting
    sorting_params = get_sorting_params(sorting_option)
    filter_aggregate = {
        'name': 'hearing',
        'type': 'terms',
        'params': {
            'field': 'hearing.targets',
            'size': 100,
            'order': sorting_params,
        }
    }
    intids = query_utility(IIntIds)
    target_ids = [intids.queryId(target) for target in context.targets]
    if target_ids:
        filter_aggregate['params']['include'] = target_ids
    
    return filter_aggregate


@adapter_config(name='hearing',
                context=(ISearchFolder, IPyAMSUserLayer),
                provides=ISearchFormRequestParams)
class SearchFormHearingRequestParams(ContextRequestAdapter):
    """Search form hearing request params"""
    
    def get_params(self):
        for hearing in self.request.params.getall('hearing'):
            yield {
                'name': 'hearing',
                'value': hearing
            }


@factory_config(ILocationFilter)
class LocationFilter(Filter):
    """Location filter"""

    filter_type = 'location'


@adapter_config(required=ILocationFilter,
                provides=IFilterAggregate)
def location_filter_aggregate(context):
    """location filter aggregate getter"""
    sorting_option = context.sorting_options
    sorting_params = get_sorting_params(sorting_option)
    return {
        'name': 'location',
        'type': 'terms',
        'params': {
            'field': 'location.departments',
            'size': 100,
            'order': sorting_params
        }
    }


@adapter_config(name='location',
                context=(ISearchFolder, IPyAMSUserLayer),
                provides=ISearchFormRequestParams)
class SearchFormLocationRequestParams(ContextRequestAdapter):
    """Search form location request params"""
    
    def get_params(self):
        for location in self.request.params.getall('location'):
            yield {
                'name': 'location',
                'value': location
            }


#
# Filters container
#

@factory_config(IFilterContainer)
class FilterContainer(BTreeOrderedContainer):
    """Filter container"""

    def add(self, obj):
        intids = get_utility(IIntIds)
        self[hex(intids.register(obj))[2:]] = obj

    def get_visible_filters(self):
        yield from filter(lambda x: x.visible, self.values())


@adapter_config(required=IFilterContainer,
                provides=IFormContextPermissionChecker)
class FilterContainerPermissionChecker(ContextAdapter):
    """Filter container permission checker"""

    edit_permission = MANAGE_TEMPLATE_PERMISSION


@adapter_config(required=IFilterContainerTarget,
                provides=IFilterContainer)
def filter_container(context):
    """Filter container adapter"""
    return get_annotation_adapter(context, FILTER_CONTAINER_ANNOTATION_KEY,
                                  IFilterContainer, name='++filter++')


@adapter_config(name='filter',
                required=IFilterContainerTarget,
                provides=ITraversable)
class FilterContainerTraverser(ContextAdapter):
    """Filter container traverser"""

    def traverse(self, name, furtherPath=None):
        container = IFilterContainer(self.context, None)
        if name:
            return container.get(name)
        return container


@adapter_config(name='filter',
                required=IFilterContainerTarget,
                provides=ISublocations)
class FilterContainerSublocations(ContextAdapter):
    """Filter container sublocations"""

    def sublocations(self):
        return IFilterContainer(self.context).values()


#
# Advanced filters portlet renderer settings
#

@factory_config(ISearchResultsPortletAdvancedFiltersRendererSettings)
@implementer(IFilterContainerTarget, IAggregatedPortletRenderer)
class SearchResultsPortletAdvancedFiltersRendererSettings(SearchResultsPortletRendererBaseSettings):
    """Search results portlet advanced filters renderer settings"""

    @property
    def filters(self):
        yield from IFilterContainer(self, {}).get_visible_filters()

    def get_labels_for_tag_ids(self, tag_data):
        intids = get_utility(IIntIds)
        result = {}
        for item in tag_data:
            try:
                tag_id = int(item['key'])
                tag_object = intids.queryObject(tag_id)
                if tag_object is not None:
                    result[tag_id] = tag_object.label
            except (KeyError, ValueError, TypeError):
                continue
        return result

    def processed_filters(self, request, aggregations):
        processed = []
        for filter in self.filters:
            processor = request.registry.queryAdapter(filter, IFilterProcessor)
            if processor is not None:
                processed_data = processor.process(request, aggregations)
                processed.append(processed_data)
        return processed

    @property
    def aggregates(self):
        return [
            IFilterAggregate(filter)
            for filter in self.filters
        ]


@adapter_config(name='advanced-filters',
                required=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/advanced-filters.pt',
                 layer=IPyAMSLayer)
@implementer(ISearchResultsView)
class SearchResultsPortletAdvancedFiltersRenderer(AggregatedSearchResultsRenderer):
    """Advanced filters search results portlet renderer"""

    label = _("Search results with advanced filters")
    weight = 65

    settings_interface = ISearchResultsPortletAdvancedFiltersRendererSettings

    def update(self):
        super().update()
        aside_sticky.need()


SORTING_MODES = {
    'alphabetical_asc': lambda x: locale.strxfrm(x['label'] or ''),
    'alphabetical_desc': lambda x: locale.strxfrm(x['label'] or ''),
    'count_asc': lambda x: (x['doc_count'], locale.strxfrm(x['label'] or '')),
    'count_desc': lambda x: (x['doc_count'], locale.strxfrm(x['label'] or ''))
}


def get_sorting_key(sorting_option):
    return SORTING_MODES.get(sorting_option, 'alphabetical_asc')


@adapter_config(required=IFilter,
                provides=IFilterProcessor)
class BaseFilterProcessor(ContextAdapter):
    """Base filter processor"""

    def process(self, request, aggregations, filter_type=None):
        filter_type = filter_type or self.context.filter_type
        current_filters = request.params.getall(filter_type)
        aggr = self.get_aggregations(aggregations[filter_type])
        initial_options = self.displayed_filter_options(aggr, self.context.displayed_entries)
        show_more = len(aggr) > self.context.displayed_entries
        placeholder = II18n(self.context).query_attribute("select_placeholder")
        additional_options = [option for index, option in enumerate(aggr) if index >= self.context.displayed_entries]
        is_expanded_condition = any(item['key'] in current_filters for item in additional_options)
        is_expanded = "in" if show_more and is_expanded_condition else ""
        translate = request.localizer.translate
        toggle_button_text = translate(_("Show less") if is_expanded == "in" else _("Show more"))
        if not placeholder:
            request = query_request()
            if request is not None:
                translate = request.localizer.translate
                placeholder = translate(_("-- No selected option --"))
        return {
            'filter': self.context,
            'filter_label': II18n(self.context).get_attribute("label"),
            'current_filters': current_filters,
            'initial_options': initial_options,
            'additional_options': additional_options,
            'all_options': aggr,
            'is_active': bool(current_filters),
            'show_more': show_more,
            'display_mode': self.context.display_mode,
            "select_placeholder": placeholder,
            "is_expanded": is_expanded,
            "toggle_button_text": toggle_button_text,
            "filter_type": filter_type
        }

    def get_aggregations(self,  aggregations):
        result = []
        for item in aggregations:
            result.append({
                'key': item.key,
                'label': item.key,
                'doc_count': item['doc_count'],
            })
        sorting_option = self.context.sorting_options
        sorting_key = get_sorting_key(sorting_option)
        if sorting_key:
            result.sort(key=sorting_key,
                        reverse=sorting_option.endswith('_desc'))
        return result

    def displayed_filter_options(self, options, num_entries_displayed):
        return options[:num_entries_displayed]


class BaseFilterAggregationProcessor(BaseFilterProcessor):
    """Base filter aggregation processor"""

    def get_aggregations(self, aggregations):
        sorting_option = self.context.sorting_options
        sorted_aggregations = self.transform_aggregation_data(aggregations)
        sorting_key = get_sorting_key(sorting_option)
        if sorting_key:
            sorted_aggregations.sort(key=sorting_key,
                                     reverse=sorting_option.endswith('_desc'))
        return sorted_aggregations

    def transform_aggregation_data(self, aggregation_data):
        intids = get_utility(IIntIds)
        result = []
        for item in aggregation_data:
            tag_id = int(item['key'])
            tag_object = intids.queryObject(tag_id)
            if tag_object:
                pictogram = None
                feature_info = IThesaurusTermFilterInfo(tag_object, None)
                if feature_info is not None:
                    pictogram = feature_info.get_pictogram()
                result.append({
                    'key': tag_object.label,
                    'label': tag_object.public_label or tag_object.label,
                    'pictogram': pictogram,
                    'doc_count': item['doc_count']
                })
    
        return result


@adapter_config(required=IContentTypeFilter,
                provides=IFilterProcessor)
class ContentTypeFilterProcessor(BaseFilterProcessor):
    """Content-type filter processor"""
    
    def process(self, request, aggregations):
        display_mode = self.context.content_type_display_mode
        result = super().process(request, aggregations, display_mode)
        return result


@adapter_config(required=ITagFilter,
                provides=IFilterProcessor)
class TagsFilterProcessor(BaseFilterAggregationProcessor):
    """Tags filter processor"""


@adapter_config(required=IThemeFilter,
                provides=IFilterProcessor)
class ThemeFilterProcessor(BaseFilterAggregationProcessor):
    """Themes filter processor"""


@adapter_config(required=ICollectionFilter,
                provides=IFilterProcessor)
class CollectionFilterProcessor(BaseFilterAggregationProcessor):
    """Collections filter processor"""


@adapter_config(required=ITargetFilter,
                provides=IFilterProcessor)
class TargetFilterProcessor(BaseFilterProcessor):
    """Targets filter processor"""

    def get_aggregations(self, aggregations):
        sorting_option = self.context.sorting_options
        sorted_aggregations = self.get_extract_labels_based_on_hearing(aggregations)
        if sorting_option != 'manual':
            sorting_key = get_sorting_key(sorting_option)
            if sorting_key:
                sorted_aggregations.sort(key=sorting_key,
                                         reverse=sorting_option.endswith('_desc'))
        else:
            intids = get_utility(IIntIds)
            target_ids = [intids.queryId(target) for target in self.context.targets]
            id_to_bucket = {
                bucket['id']: bucket
                for bucket in sorted_aggregations
            }
            sorted_aggregations = [
                id_to_bucket[target_id]
                for target_id in target_ids
                if target_id in id_to_bucket
            ]
        return sorted_aggregations

    def get_extract_labels_based_on_hearing(self, target_data):
        intids = get_utility(IIntIds)
        result = []
        for item in target_data:
            target_id = int(item['key'])
            target_object = intids.queryObject(target_id)
            if target_object:
                facets_label = II18n(target_object).query_attributes_in_order(('facets_label', 'title'))
                result.append({
                    'key': facets_label,
                    'label': facets_label,
                    'doc_count': item['doc_count'],
                    'id': target_id
                })
        return result


@adapter_config(required=ILocationFilter,
                provides=IFilterProcessor)
class LocationFilterProcessor(BaseFilterProcessor):
    """Locations filter processor"""

    def get_aggregations(self, aggregations):
        sorting_option = self.context.sorting_options
        sorted_aggregations = self.get_extract_labels_based_on_department_code(aggregations)
        sorting_key = get_sorting_key(sorting_option)
        if sorting_key:
            sorted_aggregations.sort(key=sorting_key,
                                     reverse=sorting_option.endswith('_desc'))
        return sorted_aggregations

    def get_extract_labels_based_on_department_code(self, departments):
        departmentsTable = get_utility(IDepartmentTable)
        result = []
        for item in departments:
            departmentCode  = item['key']
            if departmentCode:
                departmentInfo = departmentsTable.get(departmentCode)
                if departmentInfo is not None:
                    result.append({
                        'key': departmentCode,
                        'label': II18n(departmentInfo).get_attribute("title"),
                        'doc_count': item['doc_count'],
                    })
        return result
