#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import builtins
import html

from zope.interface import Interface, implementer
from zope.intid import IIntIds
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing import IHearingInfo
from onf_website.features.search.skin.interfaces import \
    ICardsWithHeadFiltersSearchResultsPortletRendererSettings, \
    IHeadFiltersSearchResultsPortletRendererSettings
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_content.component.theme.interfaces import ICollectionsManager, ITagsManager
from pyams_content.features.search.interfaces import ISearchFolder
from pyams_content.features.search.portlet.interfaces import IAggregatedPortletRenderer, \
    ISearchResultsPortletSettings
from pyams_content.shared.common import CONTENT_TYPES
from pyams_content.shared.common.interfaces import ISharedTool, IWfSharedContent
from pyams_content.shared.common.interfaces.types import ITypedDataManager, IWfTypedSharedContent
from pyams_content.shared.common.portlet.interfaces import ISharedContentHeaderPortletSettings
from pyams_default_theme.features.search.portlet import SearchResultsPortletBaseRenderer, \
    SearchResultsPortletDefaultRenderer, SearchResultsPortletPanelsRenderer, \
    SearchResultsPortletRendererBaseSettings, WfSharedContentSearchResultPanelRenderer, \
    WfSharedContentSearchResultRenderer
from pyams_default_theme.interfaces import IContentTag, ISearchResultsView
from pyams_default_theme.shared.common.interfaces import ISharedContentHeadViewletManager
from pyams_default_theme.shared.common.portlet.head import SharedContentHeaderPortletRenderer
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_skin.interfaces.viewlet import IBreadcrumbs
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.breadcrumb import BreadcrumbsAdapter, \
    BreadcrumbsAdapter as BaseBreadcrumbsAdapter
from pyams_template.template import override_template, template_config
from pyams_thesaurus.interfaces.thesaurus import IThesaurus
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_local_registry, query_utility
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_workflow.interfaces import IWorkflowPublicationInfo

__docformat__ = 'restructuredtext'

from onf_website import _


#
# Search folder head and specificities
#

@adapter_config(context=(ISearchFolder, IONFBaseLayer, Interface,
                         ISharedContentHeaderPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/head.pt', layer=IONFBaseLayer)
class SearchFolderHeaderPortletRenderer(SharedContentHeaderPortletRenderer):
    """Search folder header portlet default renderer"""

    @property
    def query_label(self):
        translate = self.request.localizer.translate
        params = self.request.params
        query = params.get('user_search')
        tags = params.getall('tag')
        collections = params.getall('collection')
        if query:
            if tags:
                if len(tags) == 1:
                    tags_label = _("(on tag “{}“)")
                else:
                    tags_label = _("(on tags “{}“)")
                return '<br />'.join((translate(_("Your results for “{}“")).format(html.escape(query)),
                                      translate(tags_label)
                                          .format(', '.join(html.escape(term) for term in tags))))
            if collections:
                if len(collections) == 1:
                    collections_label = _("(on collection “{}“)")
                else:
                    collections_label = _("(on collections “{}“)")
                return '<br />'.join((translate(_("Your results for “{}“")).format(html.escape(query)),
                                      translate(collections_label)
                                          .format(', '.join(html.escape(term) for term in collections))))
            return translate(_("Your results for “{}“")) \
                .format(html.escape(query))
        if tags:
            return translate(_("Your results for “{}“")) \
                .format(html.escape(', '.join(html.escape(term) for term in tags)))
        if collections:
            if len(collections) == 1:
                collections_label = _("Your results for collection “{}“)")
            else:
                collections_label = _("Your results for collections “{}“)")
            return translate(collections_label) \
                .format(', '.join(html.escape(term) for term in collections))
        return html.escape(II18n(self.context).query_attribute('title',
                                                               request=self.request))

    def get_term(self, thesaurus_name, param_name):
        if not thesaurus_name:
            return None
        thesaurus = query_utility(IThesaurus, name=thesaurus_name)
        if thesaurus is None:
            return None
        term = thesaurus.terms.get(self.request.params.get(param_name))
        if term is None:
            return None
        return term.public_label or term.label

    @property
    def tag_label(self):
        """Search tag label"""
        settings = ITagsManager(self.request.root, None)
        if settings is None:
            return None
        return self.get_term(settings.thesaurus_name, 'tag')

    @property
    def collection_label(self):
        """Search collection label"""
        settings = ICollectionsManager(self.request.root, None)
        if settings is None:
            return None
        return self.get_term(settings.thesaurus_name, 'collection')


@viewlet_config(name='search-folder-head', context=ISearchFolder, layer=IONFBaseLayer,
                view=Interface, manager=ISharedContentHeadViewletManager, weight=1)
@template_config(template='templates/head-specificities.pt', layer=IONFBaseLayer)
class SearchFolderHeadViewlet(Viewlet):
    """Search folder head specificities viewlet"""


#
# Base aggregated search results renderer
#

class AggregatedSearchResultsRenderer(SearchResultsPortletBaseRenderer):
    """Base filters aggregations renderer"""

    intids = None

    def update(self):
        super().update()
        self.intids = query_utility(IIntIds)

    def get_contenttype_label(self, content_type):
        """Get label of given content type"""
        translate = self.request.localizer.translate
        return translate(CONTENT_TYPES[content_type].content_name)

    def get_datatype_label(self, data_type):
        """Get label of given data type"""
        registry = get_local_registry()
        for tool in registry.getAllUtilitiesRegisteredFor(ISharedTool):
            manager = ITypedDataManager(tool, None)
            if manager is None:
                continue
            type = manager.get(data_type)
            if type is not None:
                return II18n(type).query_attribute('label', request=self.request)
        return None

    def get_sorted_aggregations(self, themes_list, aggregations, alpha_order):
        themes_ids = [
            str(self.intids.queryId(theme))
            for theme in sorted(themes_list,
                                key=lambda x: x.label)
        ]
        if alpha_order:
            yield from sorted(builtins.filter(lambda x: x.key in themes_ids, aggregations),
                              key=lambda x: themes_ids.index(x.key))
        else:
            yield from sorted(builtins.filter(lambda x: x.key in themes_ids, aggregations),
                              key=lambda x: x.doc_count, reverse=True)

    def get_theme(self, theme_id):
        return self.intids.queryObject(int(theme_id))


#
# Custom head-filtered portlet renderer
#

@factory_config(provided=IHeadFiltersSearchResultsPortletRendererSettings)
@implementer(IAggregatedPortletRenderer)
class HeadFiltersSearchResultsPortletRendererSettings(SearchResultsPortletRendererBaseSettings):
    """Head filters search results portlet renderer settings"""

    placeholder = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['placeholder'])
    content_types_filter = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['content_types_filter'])
    _content_types_label = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['content_types_label'])
    data_types_filter = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['data_types_filter'])
    ignored_data_types = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['ignored_data_types'])
    _data_types_label = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['data_types_label'])
    themes_filter_1 = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['themes_filter_1'])
    themes_filter_1_label = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['themes_filter_1_label'])
    themes_filter_1_alpha = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['themes_filter_1_alpha'])
    themes_filter_2 = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['themes_filter_2'])
    themes_filter_2_label = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['themes_filter_2_label'])
    themes_filter_2_alpha = FieldProperty(
        IHeadFiltersSearchResultsPortletRendererSettings['themes_filter_2_alpha'])

    @property
    def content_types_label(self):
        label = self._content_types_label
        if not label:
            label = {
                'en': "Content types",
                'fr': "Types de contenus"
            }
        return label

    @content_types_label.setter
    def content_types_label(self, value):
        self._content_types_label = value

    @property
    def data_types_label(self):
        label = self._data_types_label
        if not label:
            label = {
                'en': "Data types",
                'fr': "Types de contenus"
            }
        return label

    @data_types_label.setter
    def data_types_label(self, value):
        self._data_types_label = value

    @property
    def aggregates(self):
        result = []
        if self.content_types_filter:
            result.append({
                'name': 'content_type',
                'type': 'terms',
                'params': {
                    'field': 'content_type',
                    'size': 100
                }
            })
        if self.data_types_filter:
            result.append({
                'name': 'data_type',
                'type': 'terms',
                'params': {
                    'field': 'data_type',
                    'size': 100
                }
            })
        if self.themes_filter_1 or self.themes_filter_2:
            result.append({
                'name': 'themes',
                'type': 'terms',
                'params': {
                    'field': 'themes.terms',
                    'size': 100
                }
            })
        return result


@adapter_config(name='head-filters',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/search-head-filters.pt', layer=IPyAMSLayer)
@implementer(ISearchResultsView)
class HeadFiltersSearchResultsPortletRenderer(AggregatedSearchResultsRenderer):
    """Head filters search results portlet renderer"""

    label = _("Search results with head filters")
    weight = 50

    settings_interface = IHeadFiltersSearchResultsPortletRendererSettings


#
# Search results with cards renderer
#

@factory_config(provided=ICardsWithHeadFiltersSearchResultsPortletRendererSettings)
class CardsWithHeadFiltersSearchResultsPortletRendererSettings(
        HeadFiltersSearchResultsPortletRendererSettings):
    """Head filters search results portlet renderer settings"""

    main_css_class = FieldProperty(
        ICardsWithHeadFiltersSearchResultsPortletRendererSettings['main_css_class'])
    inner_css_class = FieldProperty(
        ICardsWithHeadFiltersSearchResultsPortletRendererSettings['inner_css_class'])
    display_tags = FieldProperty(
        ICardsWithHeadFiltersSearchResultsPortletRendererSettings['display_tags'])


@adapter_config(name='cards-head-filters',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/search-cards-filters.pt', layer=IPyAMSLayer)
@implementer(ISearchResultsView)
class CardsWithHeadFiltersSearchResultsPortletRenderer(HeadFiltersSearchResultsPortletRenderer):
    """Cards with head filters search results portlet renderer"""

    label = _("Search results cards with head filters")
    weight = 60

    settings_interface = ICardsWithHeadFiltersSearchResultsPortletRendererSettings
    default_page_length = 12


#
# Search folder breadcrumbs
#

@adapter_config(context=(ISearchFolder, IONFBaseLayer, Interface), provides=IBreadcrumbs)
class SearchFolderBreadcrumbsAdapter(BreadcrumbsAdapter):
    """Search folder breadcrumbs adapter

    Use default breadcrumbs adapter.
    """


#
# Search folder tags
#

@adapter_config(context=(ISearchFolder, IONFBaseLayer),
                provides=IContentTag)
def search_folder_tag_adapter(context, request):
    """Search folder tag adapter"""
    return None


#
# Custom renderers templates
#

override_template(context=SearchResultsPortletDefaultRenderer,
                  template='templates/search-results.pt', layer=IONFBaseLayer)

override_template(context=WfSharedContentSearchResultRenderer,
                  template='templates/search-result.pt', layer=IONFBaseLayer)

override_template(name='pdcm',
                  context=WfSharedContentSearchResultRenderer,
                  template='templates/search-result.pt', layer=IONFBaseLayer)


override_template(context=SearchResultsPortletPanelsRenderer,
                  template='templates/search-panels.pt', layer=IONFBaseLayer)

override_template(context=WfSharedContentSearchResultPanelRenderer,
                  template='templates/search-panel.pt', layer=IONFBaseLayer)


#
# Custom breadcrumbs adapter
#

@adapter_config(context=(IWfSharedContent, IONFBaseLayer, ISearchResultsView),
                provides=IBreadcrumbs)
@adapter_config(context=(IWfTypedSharedContent, IONFBaseLayer, ISearchResultsView),
                provides=IBreadcrumbs)
class BreadcrumbsAdapter(BaseBreadcrumbsAdapter):
    """Custom search results breadcrumbs adapter"""

    @property
    def items(self):
        hearing = IHearingInfo(self.context, None)
        if hearing is not None:
            source = hearing.get_source_folder()
            if source is None:
                source = hearing.get_source_site()
            if source is not None:
                publication_info = IWorkflowPublicationInfo(source)
                if publication_info.is_visible(self.request):
                    yield from self.get_items(source)
