#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from pyramid.decorator import reify
from zope.contentprovider.interfaces import IContentProvider
from zope.interface import Interface
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from onf_website.features.menu.skin.interfaces import IONFCarouselNavigationPortletRendererSettings, \
    IONFCatalogNavigationPortletRendererSettings, IONFExternalNavigationPortletRendererSettings, \
    IONFFramedNavigationPortletRendererSettings, IONFLeisuresNavigationPortletRendererSettings, \
    IONFMobileApplicationsNavigationPortletRendererSettings, IONFNavigationPortletRendererSettings, \
    IONFNavigationPortletRendererSettingsWithBackground, IONFNavigationPortletRendererSettingsWithSubtitle, \
    IONFTopicNavigationPortletRendererSettings
from onf_website.reference.forest.model.interfaces.foret import StatutForetException
from onf_website.shared.forest import IWfForest
from onf_website.skin.public import aside_sticky, owl_carousel_init, owl_navigation
from pyams_content.component.association.interfaces import IAssociationInfo
from pyams_content.component.links import IInternalLink, InternalReferenceMixin
from pyams_content.features.menu.portlet.navigation.interfaces.simple import ISimpleNavigationPortletSettings
from pyams_content.features.search.interfaces import IContextUserSearchSettings
from pyams_content.root import ISiteRootConfiguration
from pyams_file.property import FileProperty
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_portal.portlet import PortletRenderer
from pyams_sequence.reference import get_reference_target
from pyams_skin.interfaces.configuration import IConfiguration
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config

from onf_website import _


#
# Generic navigation portlet settings
#

@factory_config(provided=IONFNavigationPortletRendererSettings)
class ONFNavigationPortletRendererSettings(Persistent, Location):
    """ONF navigation portlet renderer settings"""

    header = FieldProperty(IONFNavigationPortletRendererSettings['header'])


@factory_config(provided=IONFNavigationPortletRendererSettingsWithSubtitle)
class ONFNavigationPortletRendererSettingsWithSubtitle(ONFNavigationPortletRendererSettings):
    """ONF navigation portlet renderer settings with subtitle"""

    subtitle = FieldProperty(IONFNavigationPortletRendererSettingsWithSubtitle['subtitle'])


@factory_config(provided=IONFNavigationPortletRendererSettingsWithBackground)
class ONFNavigationPortletRendererSettingsWithBackground(ONFNavigationPortletRendererSettings):
    """ONF navigation portlet renderer settings with background images"""

    top_image = FileProperty(
        IONFNavigationPortletRendererSettingsWithBackground['top_image'])
    bottom_left_image = FileProperty(
        IONFNavigationPortletRendererSettingsWithBackground['bottom_left_image'])
    bottom_right_image = FileProperty(
        IONFNavigationPortletRendererSettingsWithBackground['bottom_right_image'])


#
# Base simple navigation portlet renderer
#

class ONFBaseSimpleNavigationPortletRenderer(PortletRenderer):
    """ONF base simple navigation portlet renderer"""

    settings_interface = IONFNavigationPortletRendererSettings

    @staticmethod
    def get_link_info(link):
        return IAssociationInfo(link)

    @reify
    def public_url(self):
        configuration = IConfiguration(self.request.root, None)
        if not ISiteRootConfiguration.providedBy(configuration):
            raise ''
        return configuration.public_url


class ONFBaseSimpleNavigationPortletRendererWithSubtitle(ONFBaseSimpleNavigationPortletRenderer):
    """ONF base simple navigation portlet renderer including subtitle"""

    settings_interface = IONFNavigationPortletRendererSettingsWithSubtitle


#
# Horizontal tabs navigation portlet renderer
#

@adapter_config(name='ONF::navigation-tabs',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-tabs.pt', layer=IPyAMSLayer)
class ONFNavigationTabsPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF horizontal tabs navigation portlet renderer"""

    label = _("ONF: Horizontal navigation tabs")
    weight = 10


#
# Horizontal tabs with background images portlet renderer
#

@adapter_config(name='ONF::navigation-tabs-with-background',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-tabs-background.pt', layer=IPyAMSLayer)
class ONFNavigationTabsWithBackgroundPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF horizaontal tabs with background portlet renderer"""

    label = _("ONF: Horizontal navigation tabs with background images")
    weight = 11

    settings_interface = IONFNavigationPortletRendererSettingsWithBackground


#
# Horizontal banner navigation portlet renderer
#

@adapter_config(name='ONF::navigation-banner',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-banner.pt', layer=IPyAMSLayer)
class ONFNavigationBannerPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF horizontal banner navigation portlet renderer"""

    label = _("ONF: Horizontal navigation banner")
    weight = 12

    settings_interface = None

    def get_tags_viewlet(self, link):
        if not IInternalLink.providedBy(link):
            return ''
        target = link.get_target()
        if target is not None:
            registry = self.request.registry
            viewlet = registry.queryMultiAdapter((target, self.request, self), IContentProvider, name='pyams.tags')
            if viewlet is not None:
                viewlet.update()
                return viewlet.render()
        return ''


#
# Horizontal navigation with background illustrations portlet renderer
#

@adapter_config(name='ONF::navigation-illustration-shortcuts',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-shortcuts.pt', layer=IPyAMSLayer)
class ONFShortcutNavigationPortletRenderer(ONFBaseSimpleNavigationPortletRendererWithSubtitle):
    """ONF simple navigation horizontal portlet renderer with tabs"""

    label = _("ONF: Horizontal links list with background illustrations")
    weight = 13


#
# Vertical illustrations portlet renderer
#

@adapter_config(name='ONF::navigation-illustration-vertical',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-illustration-vertical.pt', layer=IPyAMSLayer)
class ONFNavigationIllustrationVerticalPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF vertical illustrations navigation portlet renderer"""

    label = _("ONF: Vertical illustrations navigation")
    weight = 14

    resources = (owl_carousel_init, owl_navigation)


#
# Accordion illustrations portlet renderer
#

@adapter_config(name='ONF::navigation-illustration-accordion',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-illustration-accordion.pt', layer=IPyAMSLayer)
class ONFNavigationIllustrationAccordionPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF accordion illustrations navigation portlet renderer"""

    label = _("ONF: Accordion of horizontal illustrations")
    weight = 15


#
# External navigation portlet renderer
#

@factory_config(provided=IONFExternalNavigationPortletRendererSettings)
class ONFExternalNavigationPortletRendererSettings(Persistent, Location):
    """ONF external navigation portlet renderer settings"""

    button_label = FieldProperty(IONFExternalNavigationPortletRendererSettings['button_label'])


@adapter_config(name='ONF::navigation-external-shortcuts',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-external-shortcuts.pt', layer=IPyAMSLayer)
class ONFExternalShortcutNavigationPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF External navigation portlet renderer"""

    label = _("ONF: External navigation links with illustrations")
    weight = 16

    settings_interface = IONFExternalNavigationPortletRendererSettings


#
# Useful links vertical navigation portlet renderer
#

@adapter_config(name='ONF::navigation-vertical-useful-links',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-vertical-useful-links.pt', layer=IPyAMSLayer)
class ONFNavigationVerticalUsefulLinksPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF Useful links vertical navigation portlet renderer"""

    label = _("ONF: Useful links vertical list")
    weight = 17

    settings_interface = None

    resources = (aside_sticky,)


#
# Info links vertical navigation portlet renderer
#

@adapter_config(name='ONF::navigation-vertical-info-links',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-vertical-info-links.pt', layer=IPyAMSLayer)
class ONFNavigationVerticalInfoLinksPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF Info links vertical navigation portlet renderer"""

    label = _("ONF: Info links button vertical list")
    weight = 18

    resources = (aside_sticky,)
    settings_interface = None


#
# Catalog navigation portlet renderer
#

@factory_config(provided=IONFCatalogNavigationPortletRendererSettings)
class ONFCatalogNavigationPortletRendererSettings(Persistent, Location, InternalReferenceMixin):
    """ONF catalog navigation portlet renderer settings"""

    teaser = FieldProperty(IONFCatalogNavigationPortletRendererSettings['teaser'])
    button_label = FieldProperty(IONFCatalogNavigationPortletRendererSettings['button_label'])
    reference = FieldProperty(IONFCatalogNavigationPortletRendererSettings['reference'])


@adapter_config(name='ONF::navigation-catalog',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-catalog.pt', layer=IPyAMSLayer)
class ONFCatalogNavigationPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF catalog navigation portlet renderer"""

    label = _("ONF: Catalog horizontal renderer")
    weight = 19

    settings_interface = IONFCatalogNavigationPortletRendererSettings


#
# Single navigation portlet renderer
#

@adapter_config(name='ONF::navigation-single-link',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-single-link.pt', layer=IPyAMSLayer)
class ONFSingleLinkNavigationPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF Single link navigation portlet renderer"""

    label = _("ONF: Single link navigation banner")
    weight = 20

    settings_interface = None


#
# Carousel navigation portlet renderer
#

@factory_config(provided=IONFCarouselNavigationPortletRendererSettings)
class ONFCarouselNavigationPortletRendererSettings(Persistent, Location):
    """ONF carousel navigation portlet renderer settings"""

    teaser = FieldProperty(IONFCarouselNavigationPortletRendererSettings['teaser'])


@adapter_config(name='ONF::navigation-carousel',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-carousel.pt', layer=IPyAMSLayer)
class ONFCarouselNavigationPortletRenderer(ONFNavigationBannerPortletRenderer):
    """ONF Carousel navigation with teaser portlet renderer"""

    label = _("ONF: Carousel horizontal navigation renderer ")
    weight = 21

    resources = (owl_carousel_init,)
    settings_interface = IONFCarouselNavigationPortletRendererSettings


#
# Grid navigation portlet renderer
#

@adapter_config(name='ONF::navigation-grid',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-grid.pt', layer=IPyAMSLayer)
class ONFGridNavigationPortletRenderer(ONFNavigationBannerPortletRenderer):
    """ONF grid navigation portlet renderer"""

    label = _("ONF: Grid navigation renderer ")
    weight = 22

    settings_interface = IONFNavigationPortletRendererSettingsWithSubtitle


#
# Internal navigation with the description of the target portlet renderer
#

@adapter_config(name='ONF::navigation-illustration-description',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-illustration-description.pt', layer=IPyAMSLayer)
class ONFIllustratedDescriptionNavigationPortletRenderer(ONFNavigationBannerPortletRenderer):
    """ONF illustrated navigation links with descriptions portlet renderer"""

    label = _("ONF: Illustrated navigation link with hover description")
    weight = 23

    settings_interface = IONFNavigationPortletRendererSettings


#
#  Uniform navigation bloc portlet renderer with pictogram
#

@adapter_config(name='ONF::navigation-uniform-bloc-picto',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-uniform-bloc-picto.pt', layer=IPyAMSLayer)
class ONFUniformBlocPictoNavigationPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF Uniform navigation bloc with pictogram"""

    label = _("ONF: Uniform navigation bloc with large pictogram")
    weight = 24

    settings_interface = None


#
# Navigation links with pictogram as horizontal banner
#

@factory_config(provided=IONFTopicNavigationPortletRendererSettings)
class ONFTopicNavigationPortletRendererSettings(Persistent, Location):
    """ONF topic navigation renderer settings"""

    button_label = FieldProperty(IONFTopicNavigationPortletRendererSettings['button_label'])


@adapter_config(name='ONF::navigation-navtopic',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-topic.pt', layer=IPyAMSLayer)
class ONFTopicNavigationPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF topic navigation bloc with pictogram"""

    label = _("ONF: Overlapped navigation banners")
    weight = 25

    settings_interface = IONFTopicNavigationPortletRendererSettings


#
# Leisures navigation links with pictogram in circle
#

@factory_config(provided=IONFLeisuresNavigationPortletRendererSettings)
class ONFLeisuresNavigationPortletRendererSettings(Persistent, Location, InternalReferenceMixin):
    """ONF leisures navigation portlet with pictograms"""

    reference = FieldProperty(IONFLeisuresNavigationPortletRendererSettings['reference'])
    button_label = FieldProperty(IONFLeisuresNavigationPortletRendererSettings['button_label'])


@adapter_config(name='ONF::navigation-pictograms-circles',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-pictograms-circles.pt', layer=IPyAMSLayer)
class ONFLeisuresNavigationPortletRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF leisures navigation portlet renderer"""

    label = _("ONF: Circular links with pictograms")
    weight = 26

    settings_interface = IONFLeisuresNavigationPortletRendererSettings


#
# Navigation links with horizontal tabs
#

@adapter_config(name='ONF::horizontal-tabs',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-tabs-horizontal.pt', layer=IPyAMSLayer)
class ONFNavigationHorizontalTabsRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF horizontal tabs renderer"""

    label = _("ONF: Simple horizontal tabs")
    weight = 30

    settings_interface = None


#
# Mobile applications navigation links
#

@factory_config(provided=IONFMobileApplicationsNavigationPortletRendererSettings)
class ONFMobileApplicationsNavigationPortletRendererSettings(ONFNavigationPortletRendererSettings):
    """ONF mobile applications navigation renderer settings"""

    background_image = FileProperty(
        IONFMobileApplicationsNavigationPortletRendererSettings['background_image'])
    illustration = FileProperty(
        IONFMobileApplicationsNavigationPortletRendererSettings['illustration'])


@adapter_config(name='ONF::mobile-applications',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-applications.pt', layer=IPyAMSLayer)
class ONFMobileApplicationsNavigationRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF mobile applications navigation renderer"""

    label = _("ONF: Mobile applications links")
    weight = 35

    settings_interface = IONFMobileApplicationsNavigationPortletRendererSettings


#
# Circular illustration
#

@adapter_config(name='ONF::illustration-circular',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-illustration-circle.pt', layer=IPyAMSLayer)
class ONFCircularIllustrationNavigationRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF illustrated navigation links with margins for Forest model"""

    label = _("ONF: Navigation links with circular illustration")
    weight = 40

    settings_interface = None


#
# Squared pictograms with margin
#

@adapter_config(name='ONF::pictograms-square-margin',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-pictograms-margin.pt', layer=IPyAMSLayer)
class ONFPictogramsWithMarginNavigationRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF illustrated navigation links without margins for Forest model"""

    label = _("ONF: Navigation links with square pictograms and margin")
    weight = 41

    settings_interface = None


#
# Exception forests label
#

@adapter_config(name='ONF::exception-label',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-foret-exception.pt', layer=IPyAMSLayer)
class ONFExceptionForestNavigationRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF navigation link with exceptional forest logo"""

    label = _("ONF: Navigation link with exceptional forest logo")
    weight = 42

    settings_interface = None

    def get_exception_status(self):
        forest = IWfForest(self.context, None)
        if forest is not None:
            foret, prop, info = forest.forest_refs[0]
            if info and (info.statut_exception == StatutForetException.c):
                return 'candidate'
        return 'exception'


#
# Framed links list
#

@factory_config(provided=IONFFramedNavigationPortletRendererSettings)
class ONFFramedNavigationPortletRendererSettings(Persistent, Location):
    """ONF framed navigation portlet settings"""

    list_reference = FieldProperty(IONFFramedNavigationPortletRendererSettings['list_reference'])
    link_label = FieldProperty(IONFFramedNavigationPortletRendererSettings['link_label'])
    intro_text = FieldProperty(IONFFramedNavigationPortletRendererSettings['intro_text'])
    frame_title = FieldProperty(IONFFramedNavigationPortletRendererSettings['frame_title'])
    frame_body = FieldProperty(IONFFramedNavigationPortletRendererSettings['frame_body'])
    include_context_settings = FieldProperty(IONFFramedNavigationPortletRendererSettings['include_context_settings'])

    def get_target(self):
        return get_reference_target(self.list_reference)


@adapter_config(name='ONF::framed-list',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-framed.pt', layer=IPyAMSLayer)
class ONFFramedNavigationRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF framed navigation list"""

    label = _("ONF: Framed list")
    weight = 43

    settings_interface = IONFFramedNavigationPortletRendererSettings

    def get_context_settings(self):
        settings = self.renderer_settings
        if not (IONFFramedNavigationPortletRendererSettings.providedBy(settings) and
                settings.include_context_settings):
            return None
        for name, adapter in self.request.registry.getAdapters((self.context, self.request,),
                                                               IContextUserSearchSettings):
            yield from adapter.get_settings()


#
# Action banner
#

@adapter_config(name='ONF::action-banner',
                context=(IPortalContext, IPyAMSLayer, Interface, ISimpleNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation-action-banner.pt', layer=IPyAMSLayer)
class ONFActionBannerNavigationRenderer(ONFBaseSimpleNavigationPortletRenderer):
    """ONF Navigation link indicating exceptional forest for Forest model"""

    label = _("ONF: Navigation link advertising for forest preservation")
    weight = 45

    settings_interface = None
