#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify

from onf_website.features.header import DefaultHeaderRendererSettings
from onf_website.features.header.interfaces import IDefaultHeaderRendererSettings
from onf_website.skin.public.layer import IONFBaseLayer
from onf_website.skin.public.root import IONFRootLayer
from pyams_content.component.association.interfaces import IAssociationInfo
from pyams_content.features.header.interfaces import IHeaderRenderer, IHeaderRendererSettings, \
    IHeaderSettings, IHeaderTarget
from pyams_content.shared.site.interfaces import ISiteManager
from pyams_default_theme.features.header.skin import BaseHeaderRenderer, IHeaderClass
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces.site import ISiteRoot
from pyams_utils.traversing import get_parent
from pyams_utils.url import get_display_context, absolute_url
from pyams_viewlet.viewlet import ViewContentProvider, contentprovider_config


__docformat__ = 'restructuredtext'

from onf_website import _


#
# Frontispiece portlet
#

@contentprovider_config(name='onf.frontispiece',
                        layer=IPyAMSLayer)
@template_config(template='templates/frontispiece.pt',
                 layer=IPyAMSLayer)
class FrontispieceContentProvider(ViewContentProvider):
    """Frontispiece content provider"""

    @reify
    def header_settings(self):
        header_target = get_parent(self.context, IHeaderTarget)
        if header_target is None:
            return None
        return IHeaderSettings(header_target, None)

    @reify
    def renderer_settings(self):
        header_settings = self.header_settings
        if header_settings is None:
            return None
        renderer_settings = header_settings.get_renderer().settings
        if (renderer_settings is None) or \
                (not IDefaultHeaderRendererSettings.providedBy(renderer_settings)):
            return None
        return renderer_settings

    @property
    def show_frontispiece(self):
        renderer_settings = self.renderer_settings
        if renderer_settings is None:
            return False
        header_settings = self.header_settings
        if (not renderer_settings.show_frontispiece) or \
                (header_settings.inherit and not renderer_settings.show_frontispiece_on_childs):
            return False
        context = self.context
        display_context = get_display_context(self.request)
        if (context is not display_context) and \
                not renderer_settings.show_frontispiece_on_shared_contents:
            return False
        return True

    @staticmethod
    def get_link_info(link):
        return IAssociationInfo(link)

    def render(self, template_name=''):
        if self.renderer_settings is None:
            return ''
        return super().render(template_name)


#
# Sites subscriptions portlet
#

@contentprovider_config(name='onf.mobilespace',
                        layer=IPyAMSLayer)
@template_config(template='templates/mobilespace.pt',
                 layer=IPyAMSLayer)
class MobileSpaceContentProvider(ViewContentProvider):
    """Mobile space content provider"""

    @reify
    def header_settings(self):
        header_target = get_parent(self.context, IHeaderTarget)
        if header_target is None:
            return None
        return IHeaderSettings(header_target, None)

    @reify
    def renderer_settings(self):
        header_settings = self.header_settings
        if header_settings is None:
            return None
        renderer_settings = header_settings.get_renderer().settings
        if (renderer_settings is None) or \
                (not IDefaultHeaderRendererSettings.providedBy(renderer_settings)):
            return None
        return renderer_settings


#
# Default header renderer
#

DEFAULT_HEADER_RENDERER_NAME = 'ONF default header'


@adapter_config(name=DEFAULT_HEADER_RENDERER_NAME,
                context=(IHeaderTarget, IPyAMSLayer),
                provides=IHeaderRenderer)
@adapter_config(name=DEFAULT_HEADER_RENDERER_NAME,
                context=(IHeaderSettings, IPyAMSLayer),
                provides=IHeaderRenderer)
@template_config(template='templates/default-header.pt',
                 layer=IPyAMSLayer)
class ONFDefaultHeaderRenderer(BaseHeaderRenderer):
    """ONF default header renderer"""

    name = DEFAULT_HEADER_RENDERER_NAME
    label = _("ONF default header")
    weight = 10

    settings_key = 'ONF::default'
    settings_interface = IDefaultHeaderRendererSettings

    @staticmethod
    def get_link_info(link):
        return IAssociationInfo(link)

    @property
    def search_target(self):
        return get_reference_target(self.settings.search_target, request=self.request)

    @property
    def contact_target(self):
        return get_reference_target(self.settings.contact_target, request=self.request)

    @property
    def experiential_engine(self):
        return get_reference_target(self.settings.experiential_engine, request=self.request)

    @property
    def parent_site_url(self):
        site = get_parent(self.request.display_context, ISiteManager)
        return absolute_url(site, self.request) if site is not None else None


@adapter_config(context=ONFDefaultHeaderRenderer,
                provides=IHeaderRendererSettings)
def default_header_renderer_settings_factory(context):
    """Default header renderer settings factory"""
    return DefaultHeaderRendererSettings()


@adapter_config(context=(ISiteRoot, IONFBaseLayer),
                provides=IHeaderClass)
@adapter_config(context=(ISiteManager, IONFRootLayer),
                provides=IHeaderClass)
def siteroot_header_class_adapter(context, request):
    """Site root header class adapter"""
    return 'wrapper-header--home'
