#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from pyramid.events import subscriber
from zope.interface import alsoProvides, implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.location import Location
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty

from onf_website.features.header.interfaces import IDefaultHeaderFrontispieceMenu, \
    IDefaultHeaderMenu, \
    IDefaultHeaderMenusContainer, IDefaultHeaderRendererSettings
from pyams_content.component.association.interfaces import ASSOCIATION_CONTAINER_KEY
from pyams_content.component.illustration import IBasicIllustrationTarget
from pyams_content.component.illustration.interfaces import IIllustration
from pyams_content.features.header.interfaces import IHeaderTarget
from pyams_content.features.menu.interfaces import IMenu, IMenuLinksContainer, \
    IMenuLinksContainerTarget, IMenusContainer, IMenusContainerTarget
from pyams_file.property import FileProperty
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.traversing import get_parent


#
# Default header renderer settings
#

DEFAULT_HEADER_FRONTISPIECE_KEY = '{}::frontispiece'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_HEADER_SUBSCRIBERS_KEY = '{}::subscribers'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_HEADER_TOP_LINKS_KEY = '{}::top_links'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_HEADER_MENUS_KEY = '{}::menus'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_HEADER_SOCIAL_LINKS_KEY = '{}::social'.format(ASSOCIATION_CONTAINER_KEY)


@implementer(IDefaultHeaderRendererSettings, IMenuLinksContainerTarget, IMenusContainerTarget)
class DefaultHeaderRendererSettings(Persistent, Location):
    """Default header renderer settings"""

    _use_context_illustration = FieldProperty(
        IDefaultHeaderRendererSettings['use_context_illustration'])
    background = FileProperty(IDefaultHeaderRendererSettings['background'])
    heading = FieldProperty(IDefaultHeaderRendererSettings['heading'])
    marianne = FileProperty(IDefaultHeaderRendererSettings['marianne'])
    logo = FileProperty(IDefaultHeaderRendererSettings['logo'])
    accessibility_menu = FieldProperty(IDefaultHeaderRendererSettings['accessibility_menu'])
    show_frontispiece = FieldProperty(IDefaultHeaderRendererSettings['show_frontispiece'])
    show_frontispiece_on_childs = FieldProperty(
        IDefaultHeaderRendererSettings['show_frontispiece_on_childs'])
    show_frontispiece_on_shared_contents = FieldProperty(
        IDefaultHeaderRendererSettings['show_frontispiece_on_shared_contents'])
    show_subscriber_menu = FieldProperty(IDefaultHeaderRendererSettings['show_subscriber_menu'])
    subscriber_menu_label = FieldProperty(
        IDefaultHeaderRendererSettings['subscriber_menu_label'])
    subscriber_menu_header = FieldProperty(
        IDefaultHeaderRendererSettings['subscriber_menu_header'])
    search_menu = FieldProperty(IDefaultHeaderRendererSettings['search_menu'])
    search_target = FieldProperty(IDefaultHeaderRendererSettings['search_target'])
    experiential_engine = FieldProperty(IDefaultHeaderRendererSettings['experiential_engine'])
    contact_menu = FieldProperty(IDefaultHeaderRendererSettings['contact_menu'])
    contact_target = FieldProperty(IDefaultHeaderRendererSettings['contact_target'])
    social_links_header = FieldProperty(IDefaultHeaderRendererSettings['social_links_header'])

    @property
    def can_apply_on_root(self):
        return True

    @property
    def can_use_context_illustration(self):
        target = get_parent(self, IHeaderTarget)
        illustration = IIllustration(target, None)
        return illustration is not None

    @property
    def use_context_illustration(self):
        return self._use_context_illustration if self.can_use_context_illustration else False

    @use_context_illustration.setter
    def use_context_illustration(self, value):
        if self.can_use_context_illustration:
            self._use_context_illustration = value

    @property
    def no_use_context_illustration(self):
        return not bool(self.use_context_illustration)

    @no_use_context_illustration.setter
    def no_use_context_illustration(self, value):
        self.use_context_illustration = not bool(value)

    @property
    def frontispiece(self):
        return get_annotation_adapter(self, DEFAULT_HEADER_FRONTISPIECE_KEY, IMenu,
                                      markers=IDefaultHeaderFrontispieceMenu, name='++ass++front')

    @property
    def subscribers(self):
        return get_annotation_adapter(self, DEFAULT_HEADER_SUBSCRIBERS_KEY, IMenu,
                                      markers=IDefaultHeaderMenu, name='++ass++subs')

    @property
    def top(self):
        return get_annotation_adapter(self, DEFAULT_HEADER_TOP_LINKS_KEY, IMenu,
                                      markers=IDefaultHeaderMenu, name='++ass++top')

    @property
    def menus(self):
        return get_annotation_adapter(self, DEFAULT_HEADER_MENUS_KEY, IMenusContainer,
                                      markers=IDefaultHeaderMenusContainer, name='++ass++menus')

    @property
    def social(self):
        return get_annotation_adapter(self, DEFAULT_HEADER_SOCIAL_LINKS_KEY, IMenu,
                                      markers=IDefaultHeaderMenu, name='++ass++social')


@adapter_config(name='front',
                context=IDefaultHeaderRendererSettings,
                provides=IMenuLinksContainer)
def default_header_frontispiece_links(context):
    """Default header frontispiece links adapter"""
    return context.frontispiece


@subscriber(IObjectAddedEvent, parent_selector=IDefaultHeaderFrontispieceMenu)
def handle_added_frontispiece_menu(event):
    alsoProvides(event.object, IBasicIllustrationTarget)


@adapter_config(name='subs',
                context=IDefaultHeaderRendererSettings,
                provides=IMenuLinksContainer)
def default_header_subscribers_links(context):
    """Default header subscribers links adapter"""
    return context.subscribers


@adapter_config(name='top',
                context=IDefaultHeaderRendererSettings,
                provides=IMenuLinksContainer)
def default_header_top_links(context):
    """Default header top links adapter"""
    return context.top


@adapter_config(name='menus',
                context=IDefaultHeaderRendererSettings,
                provides=IMenusContainer)
def default_header_menus(context):
    """Default header middle links adapter"""
    return context.menus


@adapter_config(name='social',
                context=IDefaultHeaderRendererSettings,
                provides=IMenuLinksContainer)
def default_header_social_links(context):
    """Default header social links adapter"""
    return context.social


@adapter_config(name='links',
                context=IDefaultHeaderRendererSettings,
                provides=ISublocations)
class DefaultHeaderRendererSettingsSublocations(ContextAdapter):
    """Default header renderer settings sub-locations adapter"""

    def sublocations(self):
        return self.context.top.values() + self.context.menus.values() + \
               self.context.social.values()
