#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.interface import implementer
from zope.location import Location
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty

from onf_website.features.footer.interfaces import IDefaultFooterMenu, \
    IDefaultFooterRendererSettings, IONFFooterV2Menu, IONFFooterV2MenusContainer, \
    IONFFooterV2RendererSettings
from pyams_content.component.association.interfaces import ASSOCIATION_CONTAINER_KEY
from pyams_content.features.menu import IMenusContainer, IMenusContainerTarget
from pyams_content.features.menu.interfaces import IMenu, IMenuLinksContainer, IMenuLinksContainerTarget
from pyams_file.property import FileProperty
from pyams_file.schema import FileField
from pyams_sequence.reference import get_reference_target
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter

#
# Default footer renderer settings
#

DEFAULT_FOOTER_SOCIAL_LINKS_KEY = '{0}::social'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_FOOTER_SITES_LINKS_KEY = '{0}::sites'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_FOOTER_RESOURCES_LINKS_KEY = '{0}::resources'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_FOOTER_CONTACTS_LINKS_KEY = '{0}::contacts'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_FOOTER_APPS_LINKS_KEY = '{0}::apps'.format(ASSOCIATION_CONTAINER_KEY)
DEFAULT_FOOTER_LEGAL_LINKS_KEY = '{0}::legal'.format(ASSOCIATION_CONTAINER_KEY)


@implementer(IDefaultFooterRendererSettings, IMenuLinksContainerTarget)
class DefaultFooterRendererSettings(Persistent, Location):
    """Default footer renderer settings"""

    social_links_header = FieldProperty(IDefaultFooterRendererSettings['social_links_header'])
    sites_links_header = FieldProperty(IDefaultFooterRendererSettings['sites_links_header'])
    resources_links_header = FieldProperty(IDefaultFooterRendererSettings['resources_links_header'])
    contacts_links_header = FieldProperty(IDefaultFooterRendererSettings['contacts_links_header'])
    apps_links_header = FieldProperty(IDefaultFooterRendererSettings['apps_links_header'])
    legal_links_signature = FieldProperty(IDefaultFooterRendererSettings['legal_links_signature'])

    @property
    def social_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_SOCIAL_LINKS_KEY, IMenu,
                                      markers=IDefaultFooterMenu, name='++ass++social')

    @property
    def sites_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_SITES_LINKS_KEY, IMenu,
                                      markers=IDefaultFooterMenu, name='++ass++sites')

    @property
    def resources_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_RESOURCES_LINKS_KEY, IMenu,
                                      markers=IDefaultFooterMenu, name='++ass++resources')

    @property
    def contacts_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_CONTACTS_LINKS_KEY, IMenu,
                                      markers=IDefaultFooterMenu, name='++ass++contacts')

    @property
    def apps_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_APPS_LINKS_KEY, IMenu,
                                      markers=IDefaultFooterMenu, name='++ass++apps')

    @property
    def legal_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_LEGAL_LINKS_KEY, IMenu,
                                      markers=IDefaultFooterMenu, name='++ass++legal')


@adapter_config(name='social',
                context=IDefaultFooterRendererSettings,
                provides=IMenuLinksContainer)
def default_footer_social_links_adapter(context):
    """Default footer social links factory"""
    return context.social_links


@adapter_config(name='sites',
                context=IDefaultFooterRendererSettings,
                provides=IMenuLinksContainer)
def default_footer_sites_links_factory(context):
    """Default footer sites links factory"""
    return context.sites_links


@adapter_config(name='resources',
                context=IDefaultFooterRendererSettings,
                provides=IMenuLinksContainer)
def default_footer_resources_links_factory(context):
    """Default footer resources links factory"""
    return context.resources_links


@adapter_config(name='contacts',
                context=IDefaultFooterRendererSettings,
                provides=IMenuLinksContainer)
def default_footer_contacts_links_factory(context):
    """Default footer contacts links factory"""
    return context.contacts_links


@adapter_config(name='apps',
                context=IDefaultFooterRendererSettings,
                provides=IMenuLinksContainer)
def default_footer_apps_links_factory(context):
    """Default footer applications links factory"""
    return context.apps_links


@adapter_config(name='legal',
                context=IDefaultFooterRendererSettings,
                provides=IMenuLinksContainer)
def default_footer_legal_links_factory(context):
    """Default footer legal links factory"""
    return context.legal_links


@adapter_config(name='links',
                context=IDefaultFooterRendererSettings,
                provides=ISublocations)
class DefaultFooterRendererSettingsSublocations(ContextAdapter):
    """Default footer renderer settings sub-locations adapter"""

    def sublocations(self):
        return self.context.social_links.values() + \
               self.context.sites_links.values() + \
               self.context.resources_links.values() + \
               self.context.contacts_links.values() + \
               self.context.apps_links.values() + \
               self.context.legal_links.values()


#
# Footer v2
#

@implementer(IONFFooterV2RendererSettings, IMenusContainerTarget, IMenuLinksContainerTarget)
class ONFFooterV2RendererSettings(Persistent, Location):
    """ONF footer v2 renderer settings"""

    logo = FileProperty(IONFFooterV2RendererSettings['logo'])

    social_links_title = FieldProperty(IONFFooterV2RendererSettings['social_links_title'])
    newsletter_title = FieldProperty(IONFFooterV2RendererSettings['newsletter_title'])
    newsletter_header = FieldProperty(IONFFooterV2RendererSettings['newsletter_header'])
    newsletter_reference = FieldProperty(IONFFooterV2RendererSettings['newsletter_reference'])
    newsletter_link_title = FieldProperty(IONFFooterV2RendererSettings['newsletter_link_title'])
    contacts_links_title = FieldProperty(IONFFooterV2RendererSettings['contacts_links_title'])
    contacts_links_header = FieldProperty(IONFFooterV2RendererSettings['contacts_links_header'])
    sites_menus_title = FieldProperty(IONFFooterV2RendererSettings['sites_menus_title'])
    apps_links_title = FieldProperty(IONFFooterV2RendererSettings['apps_links_title'])
    legal_links_signature = FieldProperty(IONFFooterV2RendererSettings['legal_links_signature'])

    @property
    def social_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_SOCIAL_LINKS_KEY, IMenu,
                                      markers=IONFFooterV2Menu, name='++ass++social')

    @property
    def newsletter_target(self):
        return get_reference_target(self.newsletter_reference)

    @property
    def contacts_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_CONTACTS_LINKS_KEY, IMenu,
                                      markers=IONFFooterV2Menu, name='++ass++contacts')

    @property
    def sites(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_SITES_LINKS_KEY, IMenusContainer,
                                      markers=IONFFooterV2MenusContainer, name='++ass++sites')

    @property
    def menus(self):
        return self.sites

    @property
    def apps_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_APPS_LINKS_KEY, IMenu,
                                      markers=IONFFooterV2Menu, name='++ass++apps')

    @property
    def legal_links(self):
        return get_annotation_adapter(self, DEFAULT_FOOTER_LEGAL_LINKS_KEY, IMenu,
                                      markers=IONFFooterV2Menu, name='++ass++legal')


@adapter_config(name='social',
                context=IONFFooterV2RendererSettings,
                provides=IMenuLinksContainer)
def footer_v2_social_links(context):
    """ONF footer v2 social links factory"""
    return context.social_links


@adapter_config(name='sites',
                context=IONFFooterV2RendererSettings,
                provides=IMenusContainer)
def footer_v2_sites_menus(context):
    """ONF footer v2 sites menus factory"""
    return context.sites


@adapter_config(name='contacts',
                context=IONFFooterV2RendererSettings,
                provides=IMenuLinksContainer)
def footer_v2_contacts_links(context):
    """ONF footer v2 contacts links factory"""
    return context.contacts_links


@adapter_config(name='apps',
                context=IONFFooterV2RendererSettings,
                provides=IMenuLinksContainer)
def footer_v2_apps_links(context):
    """ONF footer v2 applications links factory"""
    return context.apps_links


@adapter_config(name='legal', context=IONFFooterV2RendererSettings,
                provides=IMenuLinksContainer)
def footer_v2_legal_links(context):
    """ONF footer v2 legal links factory"""
    return context.legal_links


@adapter_config(name='links',
                context=IONFFooterV2RendererSettings,
                provides=ISublocations)
class ONFFooterV2RendererSettingsSublocations(ContextAdapter):
    """ONF footer v2 renderer settings sub-locations adapter"""

    def sublocations(self):
        return self.context.social_links.values() + \
               self.context.contacts_links.values() + \
               self.context.sites.values() + \
               self.context.apps_links.values() + \
               self.context.legal_links.values()
