#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from cornice import Service
from cornice.validators import colander_validator
from pyramid.httpexceptions import HTTPBadRequest, HTTPForbidden, HTTPNotFound, HTTPOk, HTTPServiceUnavailable, \
    HTTPUnauthorized
from pyramid.security import Authenticated

from onf_website.features.firebase import IFirebaseSubscriptions
from onf_website.features.firebase.api.interfaces import REST_SUBSCRIPTIONS_ROUTE
from onf_website.features.firebase.api.schema import SubscriptionsDeleteQuery, SubscriptionsDeleteQueryBody, \
    SubscriptionsDeleteResponse, SubscriptionsGetterQuery, SubscriptionsGetterRequest, SubscriptionsGetterResponse, \
    SubscriptionsPatchQueryBody, SubscriptionsPatchResponse, SubscriptionsSetterQuery, SubscriptionsSetterResponse
from onf_website.shared.forest import IForestManager
from pyams_content.interfaces import USE_PUBLIC_API_PERMISSION
from pyams_content.shared.common import CONTENT_TYPES
from pyams_security.rest import check_cors_origin, set_cors_headers
from pyams_utils.registry import query_utility
from pyams_utils.rest import STATUS, http_error, rest_responses


#
# Subscriptions service
#

subscribe_service = Service(name=REST_SUBSCRIPTIONS_ROUTE,
                            pyramid_route=REST_SUBSCRIPTIONS_ROUTE,
                            description='ONF website API for Firebase contents subscriptions')


@subscribe_service.options(validators=(check_cors_origin, set_cors_headers))
def subscribe_options(request):
    return ''


subscribe_getter_responses = rest_responses.copy()
subscribe_getter_responses[HTTPOk.code] = SubscriptionsGetterResponse(
    description="ONF subscriptions getter service")


@subscribe_service.get(schema=SubscriptionsGetterRequest(),
                       validators=(check_cors_origin, colander_validator, set_cors_headers),
                       response_schemas=subscribe_getter_responses)
def subscribe_getter(request):
    """Subscriptions service getter"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    forests = query_utility(IForestManager)
    if forests is None:
        return http_error(request, HTTPServiceUnavailable)
    if not request.has_permission(USE_PUBLIC_API_PERMISSION, context=forests):
        return http_error(request, HTTPForbidden)
    params = request.validated.get('querystring', {})
    firebase_id = params.get('firebase_id')
    if not firebase_id:
        return http_error(request, HTTPBadRequest, "Missing Firebase ID")
    subscriptions = IFirebaseSubscriptions(request.root, None)
    if subscriptions is None:
        return http_error(request, HTTPServiceUnavailable)
    subscription = subscriptions.get(firebase_id)
    if subscription is None:
        return http_error(request, HTTPNotFound)
    content_type = params.get('content_type', '').lower()
    content_types = {content_type} if content_type else set(subscription.content_types.keys())
    return {
        'status': STATUS.SUCCESS.value,
        'subscriptions': [
            {
                'content_type': content_type,
                'forest_ids': list(subscription.content_types.get(content_type, ()))
            }
            for content_type in content_types
        ]
    }


subscribe_setter_responses = rest_responses.copy()
subscribe_setter_responses[HTTPOk.code] = SubscriptionsSetterResponse(
    description="ONF subscriptions setter service")


@subscribe_service.post(schema=SubscriptionsSetterQuery(),
                        validators=(check_cors_origin, colander_validator, set_cors_headers),
                        require_csrf=False,
                        response_schemas=subscribe_setter_responses)
def subscribe_setter(request):
    """Subscriptions service setter"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    forests = query_utility(IForestManager)
    if forests is None:
        return http_error(request, HTTPServiceUnavailable)
    if not request.has_permission(USE_PUBLIC_API_PERMISSION, context=forests):
        return http_error(request, HTTPForbidden)
    subscriptions = IFirebaseSubscriptions(request.root, None)
    if subscriptions is None:
        return http_error(request, HTTPServiceUnavailable)
    params = request.validated.get('body', {})
    firebase_id = params.get('firebase_id')
    if not firebase_id:
        return http_error(request, HTTPBadRequest, "Missing Firebase ID")
    content_type = params.get('content_type', '').lower()
    forest_ids = params.get('forest_ids')
    if not (content_type and forest_ids):
        return http_error(request, HTTPBadRequest, "Missing content type or forests IDs")
    if content_type not in CONTENT_TYPES.keys():
        return http_error(request, HTTPBadRequest, "Unknown content type")
    subscriptions.add_subscriptions(firebase_id, content_type, forest_ids)
    subscription = subscriptions[firebase_id]
    return {
        'status': STATUS.SUCCESS.value,
        'subscriptions': [{
            'content_type': content_type,
            'forest_ids': list(subscription.content_types.get(content_type, ()))
        } for content_type in subscription.content_types.keys()]
    }


subscribe_patch_responses = rest_responses.copy()
subscribe_patch_responses[HTTPOk.code] = SubscriptionsPatchResponse(
    description="ONF subscriptions patch service")


@subscribe_service.patch(schema=SubscriptionsPatchQueryBody(),
                         validators=(check_cors_origin, colander_validator, set_cors_headers),
                         require_csrf=False,
                         response_schemas=subscribe_patch_responses)
def subscribe_patch(request):
    """Subscriptions service patch"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    subscriptions = IFirebaseSubscriptions(request.root, None)
    if subscriptions is None:
        return http_error(request, HTTPServiceUnavailable)
    params = request.validated.get('body', {})
    old_firebase_id = params.get('old_firebase_id')
    new_firebase_id = params.get('new_firebase_id')
    if (old_firebase_id == new_firebase_id) or not (old_firebase_id and new_firebase_id):
        return http_error(request, HTTPBadRequest)
    subscription = subscriptions.update_firebase_id(old_firebase_id, new_firebase_id)
    if subscription is None:
        return {
            'status': STATUS.SUCCESS.value,
            'subscriptions': []
        }
    return {
        'status': STATUS.SUCCESS.value,
        'subscriptions': [{
            'content_type': content_type,
            'forest_ids': list(subscription.content_types.get(content_type, ()))
        } for content_type in subscription.content_types.keys()]
    }


subscribe_delete_responses = rest_responses.copy()
subscribe_delete_responses[HTTPOk.code] = SubscriptionsDeleteResponse(
    description="ONF subscriptions delete service")


@subscribe_service.delete(schema=SubscriptionsDeleteQueryBody(),
                          validators=(check_cors_origin, colander_validator, set_cors_headers),
                          require_csrf=False,
                          response_schemas=subscribe_delete_responses)
def subscribe_delete(request):
    """Subscriptions service delete"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    forests = query_utility(IForestManager)
    if forests is None:
        return http_error(request, HTTPServiceUnavailable)
    if not request.has_permission(USE_PUBLIC_API_PERMISSION, context=forests):
        return http_error(request, HTTPForbidden)
    subscriptions = IFirebaseSubscriptions(request.root, None)
    if subscriptions is None:
        return http_error(request, HTTPServiceUnavailable)
    params = request.validated.get('body', {})
    firebase_id = params.get('firebase_id')
    if not firebase_id:
        return http_error(request, HTTPBadRequest, "Missing Firebase ID")
    content_type = params.get('content_type', '').lower()
    forest_ids = params.get('forest_ids')
    subscriptions.drop_subscriptions(firebase_id, content_type, forest_ids)
    subscription = subscriptions.get(firebase_id)
    if subscription is None:
        return {
            'status': STATUS.SUCCESS.value,
            'subscriptions': []
        }

    return {
        'status': STATUS.SUCCESS.value,
        'subscriptions': [{
            'content_type': content_type,
            'forest_ids': list(subscription.content_types.get(content_type, ()))
        } for content_type in subscription.content_types.keys()]
    }
