#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
from persistent import Persistent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.verbatim.skin.interfaces import IONFVerbatimParagraphRendererSettings, \
    ONF_VERBATIM_PARAGRAPH_RENDERER_SETTINGS_KEY
from pyams_content.component.verbatim.interfaces import IVerbatimParagraph
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_default_theme.component.verbatim.paragraph import DefaultVerbatimParagraphRenderer, \
    LeftVerbatimParagraphRenderer, RightVerbatimParagraphRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter


__docformat__ = 'restructuredtext'

from onf_website import _


#
# Verbatim paragraph default renderer
#
from pyams_utils.factory import factory_config


@factory_config(provided=IONFVerbatimParagraphRendererSettings)
class ONFVerbatimParagraphRendererSettings(Persistent, Location):
    """Verbatim paragraph renderer settings"""

    xs_horizontal_padding = FieldProperty(
        IONFVerbatimParagraphRendererSettings['xs_horizontal_padding'])
    sm_horizontal_padding = FieldProperty(
        IONFVerbatimParagraphRendererSettings['sm_horizontal_padding'])
    md_horizontal_padding = FieldProperty(
        IONFVerbatimParagraphRendererSettings['md_horizontal_padding'])
    lg_horizontal_padding = FieldProperty(
        IONFVerbatimParagraphRendererSettings['lg_horizontal_padding'])

    def has_padding(self):
        return self.lg_horizontal_padding or \
               self.md_horizontal_padding or \
               self.sm_horizontal_padding or \
               self.xs_horizontal_padding

    def get_padding(self):
        result = []
        for device in ('xs', 'sm', 'md', 'lg'):
            padding = getattr(self, '{}_horizontal_padding'.format(device))
            if padding is not None:
                result.append('col-{0}-{1} col-{0}-offset-{2}'.format(device,
                                                                      12 - (padding * 2),
                                                                      padding))
        return ' '.join(result)


@adapter_config(context=IVerbatimParagraph,
                provides=IONFVerbatimParagraphRendererSettings)
def verbatim_paragraph_renderer_settings_factory(context):
    """Frame text paragraph renderer settings factory"""
    return get_annotation_adapter(context, ONF_VERBATIM_PARAGRAPH_RENDERER_SETTINGS_KEY,
                                  IONFVerbatimParagraphRendererSettings)


@adapter_config(name='default',
                context=(IVerbatimParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/verbatim-default.pt', layer=IPyAMSLayer)
class ONFDefaultVerbatimParagraphRenderer(DefaultVerbatimParagraphRenderer):
    """Verbatim paragraph default renderer"""

    label = _("ONF: Default verbatim renderer")
    weight = 1

    settings_interface = IONFVerbatimParagraphRendererSettings


@adapter_config(name='left',
                context=(IVerbatimParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/verbatim-left.pt', layer=IPyAMSLayer)
class ONFLeftVerbatimParagraphRenderer(LeftVerbatimParagraphRenderer):
    """Verbatim paragraph default renderer"""

    label = _("ONF: Left verbatim renderer")
    weight = 2


@adapter_config(name='right',
                context=(IVerbatimParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/verbatim-right.pt', layer=IPyAMSLayer)
class ONFRightVerbatimParagraphRenderer(RightVerbatimParagraphRenderer):
    """Verbatim paragraph default renderer"""

    label = _("ONF: Right verbatim renderer")
    weight = 3
