#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from copy import copy

from transaction.interfaces import ITransactionManager
from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from onf_website.component.paragraph.client import ClientParagraph
from onf_website.component.paragraph.interfaces.client import CLIENT_PARAGRAPH_TYPE, IClientParagraph
from pyams_content.component.paragraph.interfaces import IBaseParagraph, IParagraphContainerTarget
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_form_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_viewlet.viewlet import viewlet_config

from onf_website import _


@viewlet_config(name='add-client-paragraph.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=600)
class ClientParagraphAddMenu(BaseParagraphAddMenu):
    """Client paragraph add menu"""

    label = _("Client card...")
    label_css_class = 'fa fa-fw ' + ClientParagraph.icon_class
    url = 'add-client-paragraph.html'
    paragraph_type = CLIENT_PARAGRAPH_TYPE


@pagelet_config(name='add-client-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-client-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class ClientParagraphAddForm(BaseParagraphAddForm):
    """Client paragraph add form"""

    legend = _("Add new client card")
    dialog_class = 'modal-large'

    content_interface = IClientParagraph

    @property
    def fields(self):
        fields = super(ClientParagraphAddForm, self).fields
        if 'title' in fields:
            f = copy(fields['title'].field)
            f.default = {
                'en': "The client",
                'fr': "Le client"
            }
            fields['title'].field = f
        if 'title_2' in fields:
            f = copy(fields['title_2'].field)
            f.default = {
                'en': "Goal achievement",
                'fr': "Objectifs de l'intervention"
            }
            fields['title_2'].field = f
        return fields


@pagelet_config(name='properties.html', context=IClientParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IClientParagraph, request_type=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class ClientParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Client paragraph properties edit form"""

    prefix = 'client_properties.'

    legend = _("Edit client card properties")
    dialog_class = 'modal-large'

    content_interface = IClientParagraph

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IBaseParagraph, ())
        if 'title' in updated:
            output.setdefault('events', []).append(get_json_paragraph_refresh_event(self.context, self.request))
        updated = changes.get(IClientParagraph, ())
        if ('photo' in updated) or ('renderer' in updated):
            # we have to commit transaction to be able to handle blobs...
            if 'photo' in updated:
                ITransactionManager(self.context).get().commit()
            output.setdefault('events', []).append(get_json_form_refresh_event(self.context, self.request,
                                                                               ClientParagraphInnerEditForm))
        return output


@adapter_config(context=(IClientParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IClientParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm)
class ClientParagraphInnerEditForm(ClientParagraphPropertiesEditForm):
    """Client paragraph inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
