#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from persistent import Persistent
from pyramid_chameleon.interfaces import IChameleonTranslate
from zope.component import queryUtility
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.paragraph.skin.interfaces.html import \
    HTML_PARAGRAPH_ACCORDION_RENDERER_SETTINGS_KEY, IHTMLParagraphAccordionRendererSettings, \
    IHTMLParagraphCustomRenderer
from pyams_content.component.association import IAssociationContainer
from pyams_content.component.association.interfaces import IAssociationInfo
from pyams_content.component.paragraph.interfaces.html import IHTMLParagraph
from pyams_content.features.renderer import IContentRenderer
from pyams_default_theme.component.paragraph.html import HTMLParagraphDefaultRenderer, \
    HTMLParagraphWithLogosRenderer, RawParagraphDefaultRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.interfaces import IPageTemplate
from pyams_template.template import override_template, template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.list import boolean_iter


__docformat__ = 'restructuredtext'

from onf_website import _


#
# Raw paragraph activity renderer
#

override_template(RawParagraphDefaultRenderer,
                  name='PARA:title-left',
                  template='templates/raw-title-left.pt', layer=IPyAMSLayer)


#
# HTML paragraph default renderer
#

@adapter_config(name='default',
                context=(IHTMLParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(name='left+circle',
                 template='templates/html-illustration-left-circle.pt', layer=IPyAMSLayer)
@template_config(name='right+circle',
                 template='templates/html-illustration-right-circle.pt', layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/html-title-left.pt', layout=IPyAMSLayer)
class ONFHTMLParagraphDefaultRenderer(HTMLParagraphDefaultRenderer):
    """ONF HTML paragraph default renderer"""

    def render(self, template_name=''):
        registry = self.request.registry
        # try custom renderer
        renderer = registry.queryMultiAdapter((self.context, self.request),
                                              IHTMLParagraphCustomRenderer)
        if renderer is not None:
            renderer.update()
            return renderer.render(template_name)
        # try custom illustration renderer
        if self.illustration.data:
            renderer = self.illustration.renderer
            template = registry.queryMultiAdapter((self, self.request),
                                                  IPageTemplate,
                                                  name=renderer)
            if template is not None:
                return template(context=self.context,
                                request=self.request,
                                view=self,
                                translate=queryUtility(IChameleonTranslate))
        return super().render(template_name)

    def get_associations(self):
        yield from IAssociationContainer(self.context).get_visible_items(self.request)

    @property
    def first_link(self):
        has_links, links = boolean_iter(self.get_associations())
        if has_links:
            link = next(links)
            return link, IAssociationInfo(link)
        return None, None


#
# HTML paragraph renderer with logos
#

override_template(HTMLParagraphWithLogosRenderer,
                  name='PARA:title-left',
                  template='templates/html-title-left-logos.pt', layer=IPyAMSLayer)


#
# HTML paragraph accordion renderer
#

@factory_config(IHTMLParagraphAccordionRendererSettings)
class HTMLParagraphAccordionRendererSettings(Persistent, Location):
    """HTML paragraph accordion renderer settings"""

    collapsed = FieldProperty(IHTMLParagraphAccordionRendererSettings['collapsed'])


@adapter_config(context=IHTMLParagraph,
                provides=IHTMLParagraphAccordionRendererSettings)
def html_paragraph_accordion_renderer_settings_factory(context):
    """HTML paragraph accordion renderer settings factory"""
    return get_annotation_adapter(context, HTML_PARAGRAPH_ACCORDION_RENDERER_SETTINGS_KEY,
                                  IHTMLParagraphAccordionRendererSettings)


@adapter_config(name='accordion',
                context=(IHTMLParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/html-accordion.pt', layer=IPyAMSLayer)
class HTMLParagraphAccordionRenderer(HTMLParagraphDefaultRenderer):
    """HTML paragraph accordion renderer"""

    label = _("Rich text accordion renderer")
    weight = 30

    settings_interface = IHTMLParagraphAccordionRendererSettings
