#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.interface import alsoProvides
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.paragraph.interfaces.client import CLIENT_PARAGRAPH_NAME, CLIENT_PARAGRAPH_RENDERERS, \
    CLIENT_PARAGRAPH_TYPE, IClientParagraph
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_file.interfaces import IImage, IResponsiveImage
from pyams_file.property import FileProperty
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config

try:
    from pyams_gis.schema import GeoPointField
except ImportError:
    have_gis = False
else:
    from pyams_gis.interfaces import WGS84
    have_gis = True


@factory_config(provided=IClientParagraph)
class ClientParagraph(BaseParagraph):
    """Client paragraph"""

    factory_name = CLIENT_PARAGRAPH_TYPE
    icon_class = 'fa-handshake-o'
    icon_hint = CLIENT_PARAGRAPH_NAME

    text_1 = FieldProperty(IClientParagraph['text_1'])

    title_2 = FieldProperty(IClientParagraph['title_2'])
    text_2 = FieldProperty(IClientParagraph['text_2'])

    logo = FieldProperty(IClientParagraph['logo'])

    name = FieldProperty(IClientParagraph['name'])
    charge = FieldProperty(IClientParagraph['charge'])
    _photo = FileProperty(IClientParagraph['photo'])
    quote = FieldProperty(IClientParagraph['quote'])

    if have_gis:
        gps_location = FieldProperty(IClientParagraph['gps_location'])

    project_start = FieldProperty(IClientParagraph['project_start'])
    project_end = FieldProperty(IClientParagraph['project_end'])

    renderer = FieldProperty(IClientParagraph['renderer'])

    @property
    def photo(self):
        return self._photo

    @photo.setter
    def photo(self, value):
        self._photo = value
        if IImage.providedBy(self._photo):
            alsoProvides(self._photo, IResponsiveImage)
            
    @photo.deleter
    def photo(self):
        del self._photo

    def get_logo(self):
        if self.logo is not None:
            return get_reference_target(self.logo)


@utility_config(name=CLIENT_PARAGRAPH_TYPE,
                provides=IParagraphFactory)
class ClientParagraphFactory(BaseParagraphFactory):
    """Client paragraph factory"""

    name = CLIENT_PARAGRAPH_NAME
    content_type = ClientParagraph
    secondary_menu = True


@adapter_config(required=IClientParagraph,
                provides=IContentChecker)
class ClientParagraphContentChecker(BaseParagraphContentChecker):
    """Client paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('charge', lang, request)
            if not value:
                field_title = translate(IClientParagraph['charge'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        attrs = ('name', 'photo', 'address')
        if have_gis:
            attrs += ('gps_location',)
        for attr in attrs:
            value = getattr(self.context, attr, None)
            if not value:
                field_title = translate(IClientParagraph[attr].title)
                output.append(translate(MISSING_VALUE).format(field=field_title))
        return output


@vocabulary_config(name=CLIENT_PARAGRAPH_RENDERERS)
class ClientParagraphRendererVocabulary(RenderersVocabulary):
    """Client paragraph renderers vocabulary"""

    content_interface = IClientParagraph


@adapter_config(required=(IClientParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONClientParagraphConverter(JSONParagraphConverter):
    """JSON client paragraph converter"""

    def convert_content(self, params=None):
        result = super().convert_content(params)
        self.get_i18n_attribute(self.context, 'text_1', params.get('lang'), result)
        self.get_i18n_attribute(self.context, 'title_2', params.get('lang'), result)
        self.get_i18n_attribute(self.context, 'text_2', params.get('lang'), result)
        self.get_attribute(self.context, 'name', result)
        self.get_i18n_attribute(self.context, 'charge', params.get('lang'), result)
        url = self.get_image_url(self.context, 'photo', result)
        if url:
            result['photo'] = url
        self.get_i18n_attribute(self.context, 'quote', params.get('lang'), result)
        if have_gis and self.context.gps_location:
            srid = params.get('srid', WGS84)
            try:
                coords = self.context.gps_location.get_coordinates(srid)
                result['gps_location'] = {
                    'longitude': float(coords[0]),
                    'latitude': float(coords[1]),
                    'srid': srid
                }
            except NotImplementedError:  # unknown SRID
                pass
        return result
