#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from pyams_content.component.map.interfaces import have_gis
if have_gis:

    from z3c.form import button
    from z3c.form.interfaces import INPUT_MODE
    from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
    from zope.interface import implementer

    from onf_website.component.map.interfaces import IMapParagraph
    from pyams_content.component.map.interfaces import IMapParagraph as IMapParagraphBase
    from pyams_content.component.map.zmi.paragraph import \
        MapParagraphAddForm as BaseMapParagraphAddForm, \
        MapParagraphPropertiesEditForm as BaseMapParagraphPropertiesEditForm
    from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget, IBaseParagraph
    from pyams_content.component.paragraph.zmi import \
        BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
        get_json_paragraph_refresh_event, IParagraphInnerEditFormButtons
    from pyams_content.component.paragraph.zmi.interfaces import IParagraphInnerEditor
    from pyams_content.features.renderer.zmi.widget import RendererFieldWidget
    from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
    from pyams_form.form import ajax_config
    from pyams_form.group import NamedWidgetsGroup
    from pyams_form.interfaces.form import IInnerForm
    from pyams_pagelet.pagelet import pagelet_config
    from pyams_skin.event import get_json_form_refresh_event
    from pyams_skin.layer import IPyAMSLayer
    from pyams_utils.adapter import adapter_config

    from onf_website import _


    @pagelet_config(name='add-map-paragraph.html',
                    context=IParagraphContainerTarget, layer=IPyAMSLayer,
                    permission=MANAGE_CONTENT_PERMISSION)
    @ajax_config(name='add-map-paragraph.json',
                 context=IParagraphContainerTarget, layer=IPyAMSLayer,
                 base=BaseParagraphAJAXAddForm)
    class MapParagraphAddForm(BaseMapParagraphAddForm):
        """Map paragraph add form"""

        content_interface = IMapParagraph

        @property
        def fields(self):
            fields = super(MapParagraphAddForm, self).fields \
                .select('title', 'no_use_context_location', 'gps_location',
                        'display_marker', 'display_coordinates', 'renderer')
            fields['no_use_context_location'].widgetFactory = SingleCheckBoxFieldWidget
            return fields

        def updateGroups(self):
            self.add_group(NamedWidgetsGroup(self, 'map_location', self.widgets,
                                             ('no_use_context_location', 'gps_location'),
                                             legend=_("Don't use context location"),
                                             css_class='inner',
                                             switch=True,
                                             checkbox_switch=True,
                                             checkbox_field=IMapParagraph['no_use_context_location']))
            self.add_group(NamedWidgetsGroup(self, 'options', self.widgets,
                                             ('display_marker', 'display_coordinates',
                                              'renderer'), ))
            super(MapParagraphAddForm, self).updateGroups()


    @pagelet_config(name='properties.html',
                    context=IMapParagraph, layer=IPyAMSLayer,
                    permission=MANAGE_CONTENT_PERMISSION)
    @ajax_config(name='properties.json',
                 context=IMapParagraph, layer=IPyAMSLayer,
                 base=BaseParagraphAJAXEditForm)
    class MapParagraphPropertiesEditForm(BaseMapParagraphPropertiesEditForm):
        """Map paragraph properties edit form"""

        content_interface = IMapParagraph

        @property
        def fields(self):
            fields = super(MapParagraphPropertiesEditForm, self).fields \
                .select('title', 'no_use_context_location', 'gps_location',
                        'display_marker', 'display_coordinates', 'renderer')
            fields['no_use_context_location'].widgetFactory = SingleCheckBoxFieldWidget
            fields['renderer'].widgetFactory = RendererFieldWidget
            return fields

        def updateGroups(self):
            self.add_group(NamedWidgetsGroup(self, 'map_location', self.widgets,
                                             ('no_use_context_location', 'gps_location'),
                                             legend=_("Don't use context location"),
                                             css_class='inner',
                                             switch=True,
                                             checkbox_switch=True,
                                             checkbox_field=IMapParagraph['no_use_context_location']))
            self.add_group(NamedWidgetsGroup(self, 'options', self.widgets,
                                             ('display_marker', 'display_coordinates',
                                              'renderer'), ))
            super(MapParagraphPropertiesEditForm, self).updateGroups()


    @adapter_config(context=(IMapParagraph, IPyAMSLayer),
                    provides=IParagraphInnerEditor)
    @ajax_config(name='inner-properties.json',
                 context=IMapParagraph, layer=IPyAMSLayer,
                 permission=MANAGE_CONTENT_PERMISSION,
                 base=BaseParagraphAJAXEditForm)
    @implementer(IInnerForm)
    class MapParagraphInnerEditForm(MapParagraphPropertiesEditForm):

        legend = None

        @property
        def buttons(self):
            if self.mode == INPUT_MODE:
                return button.Buttons(IParagraphInnerEditFormButtons)
            return button.Buttons()

        def get_ajax_output(self, changes):
            output = super(self.__class__, self).get_ajax_output(changes)
            if 'title' in changes.get(IBaseParagraph, ()):
                output.setdefault('events', []).append(
                    get_json_paragraph_refresh_event(self.context, self.request))
            if 'renderer' in changes.get(IMapParagraphBase, ()):
                output.setdefault('events', []).append(
                    get_json_form_refresh_event(self.context, self.request,
                                                MapParagraphInnerEditForm))
            return output
