#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from pyams_content.component.map.interfaces import have_gis
if have_gis:

    from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
    from z3c.form.field import Fields
    from zope.interface import Interface

    from onf_website.component.map.portlet.interfaces import IMapPortletSettings
    from onf_website.component.map.portlet.skin.interfaces import \
        IMapPortletDefaultRendererSettings, IONFMapPortletHalfWidthRendererSettings
    from pyams_content.component.map.portlet.zmi import MapPortletEditor, \
        MapPortletSettingsPropertiesEditor as BaseMapPortletSettingsPropertiesEditor
    from pyams_default_theme.component.map.portlet.zmi import \
        MapPortletDefaultRendererSettingsFormManager as BaseMapPortletDefaultRendererSettingsFormManager
    from pyams_form.group import NamedWidgetsGroup
    from pyams_form.interfaces.form import IFormManager, IInnerTabForm
    from pyams_portal.zmi.portlet import PortletRendererPropertiesEditForm
    from pyams_portal.zmi.widget import PortletRendererFieldWidget
    from pyams_skin.layer import IPyAMSLayer
    from pyams_utils.adapter import adapter_config
    from pyams_zmi.form import AdminDialogEditForm

    from onf_website import _


    @adapter_config(name='properties',
                    context=(Interface, IPyAMSLayer, MapPortletEditor),
                    provides=IInnerTabForm)
    class MapPortletSettingsPropertiesEditor(BaseMapPortletSettingsPropertiesEditor):
        """Map portlet settings properties editor"""

        @property
        def fields(self):
            fields = Fields(IMapPortletSettings).select('title', 'no_use_context_location',
                                                        'gps_location', 'display_marker',
                                                        'display_coordinates', 'renderer')
            fields['no_use_context_location'].widgetFactory = SingleCheckBoxFieldWidget
            fields['renderer'].widgetFactory = PortletRendererFieldWidget
            return fields

        def updateGroups(self):
            self.add_group(NamedWidgetsGroup(self, 'map_location', self.widgets,
                                             ('no_use_context_location', 'gps_location'),
                                             legend=_("Don't use context location"),
                                             css_class='inner',
                                             switch=True,
                                             checkbox_switch=True,
                                             checkbox_field=IMapPortletSettings[
                                                 'no_use_context_location']))
            self.add_group(NamedWidgetsGroup(self, 'options', self.widgets,
                                             ('display_marker', 'display_coordinates',
                                              'renderer'), ))
            super().updateGroups()


    @adapter_config(context=(IMapPortletDefaultRendererSettings, IPyAMSLayer,
                             PortletRendererPropertiesEditForm),
                    provides=IFormManager)
    class MapPortletDefaultRendererSettingsFormManager(
            BaseMapPortletDefaultRendererSettingsFormManager):
        """Map paragraph default renderer settings form manager"""

        def getFields(self):
            fields = Fields(IMapPortletDefaultRendererSettings).omit(
                'use_default_map_configuration')
            fields['no_use_default_map_configuration'].widgetFactory = SingleCheckBoxFieldWidget
            fields['display_context_forests'].widgetFactory = SingleCheckBoxFieldWidget
            return fields

        def updateGroups(self):
            view = self.view
            view.add_group(NamedWidgetsGroup(view, 'configuration', view.widgets,
                                             list(view.fields.keys())[:-3],  # omit forests settings
                                             legend=_("Don't use default map configuration"),
                                             css_class='inner',
                                             switch=True,
                                             checkbox_switch=True,
                                             checkbox_field=IMapPortletDefaultRendererSettings[
                                                 'no_use_default_map_configuration']))
            view.add_group(NamedWidgetsGroup(view, 'options', view.widgets,
                                             ('display_context_forests', 'set_bounds_to_forests'),
                                             legend=_("Display location forests"),
                                             css_class='inner',
                                             switch=True,
                                             checkbox_switch=True,
                                             checkbox_field=IMapPortletDefaultRendererSettings[
                                                 'display_context_forests']))
            AdminDialogEditForm.updateGroups(view)


    @adapter_config(context=(IONFMapPortletHalfWidthRendererSettings, IPyAMSLayer,
                             PortletRendererPropertiesEditForm),
                    provides=IFormManager)
    class ONFMapPortletHalfWidthRendererSettingsFormManager(
            MapPortletDefaultRendererSettingsFormManager):
        """ONF map portlet half-width renderer settings form manager"""

        def getFields(self):
            fields = Fields(IONFMapPortletHalfWidthRendererSettings).omit(
                'use_default_map_configuration')
            fields['no_use_default_map_configuration'].widgetFactory = SingleCheckBoxFieldWidget
            fields['display_context_forests'].widgetFactory = SingleCheckBoxFieldWidget
            return fields

        def updateGroups(self):
            view = self.view
            view.add_group(NamedWidgetsGroup(view, 'configuration', view.widgets,
                                             list(view.fields.keys())[:-5],  # omit forests settings
                                             legend=_("Don't use default map configuration"),
                                             css_class='inner',
                                             switch=True,
                                             checkbox_switch=True,
                                             checkbox_field=IMapPortletDefaultRendererSettings[
                                                 'no_use_default_map_configuration']))
            view.add_group(NamedWidgetsGroup(view, 'options', view.widgets,
                                             ('display_context_forests', 'set_bounds_to_forests'),
                                             legend=_("Display location forests"),
                                             css_class='inner',
                                             switch=True,
                                             checkbox_switch=True,
                                             checkbox_field=IMapPortletDefaultRendererSettings[
                                                 'display_context_forests']))
            AdminDialogEditForm.updateGroups(view)
