#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.exceptions import NotFound
from pyramid.view import view_config
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues

from onf_website.component.location.interfaces import ILocationManagerSettings, ILocationManagerTarget, \
    LOCATION_SETTINGS, LOCATION_SETTINGS_KEYS
from onf_website.component.location.manager import LocationSetting
from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp, IWidgetsSuffixViewletsManager
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import AttributeSwitcherColumn, BaseTable, I18nColumn, TableViewlet, VisibilitySwitcherColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogDisplayForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer

__docformat__ = 'restructuredtext'

from onf_website import _


@viewlet_config(name='location-settings.menu', context=ILocationManagerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION, weight=710)
class LocationManagerSettingsMenu(MenuItem):
    """Location manager settings menu"""

    label = _("Location settings...")
    icon_class = 'fa-map-marker'
    url = 'location-settings.html'
    modal_target = True


@pagelet_config(name='location-settings.html', context=ILocationManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
class LocationManagerSettingsView(AdminDialogDisplayForm):
    """Location manager settings view"""

    legend = _("Set access to location settings for this tool")


class LocationManagerSettingsTable(BaseTable):
    """Location manager settings table"""

    id = 'location_manager_table'
    title = _("Location settings")
    hide_toolbar = True
    cssClasses = {'table': 'table table-bordered table-striped table-hover table-tight datatable'}
    sortOn = None

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(self.context, self.request),
            'data-ams-datatable-global-filter': 'false',
            'data-ams-datatable-info': 'false',
            'data-ams-datatable-sort': 'false',
            'data-ams-datatable-sdom': "t",
            'data-ams-unfolded-icon-on': 'fa fa-fw fa-folder-open-o',
            'data-ams-unfolded-icon-off': 'fa fa-fw fa-folder-o'
        })
        attributes.setdefault('tr', {}).update({
            'data-ams-element-name': lambda x, col: x.name
        })
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target,
            'data-ams-switcher-attribute-name': self.get_switcher_attribute
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-location-setting-visibility.json'
        elif column.__name__ == 'fold-unfold':
            return 'switch-location-setting-folding.json'

    @staticmethod
    def get_switcher_attribute(element, column):
        if column.__name__ == 'show-hide':
            return 'visible'
        elif column.__name__ == 'fold-unfold':
            return 'unfolded'


@adapter_config(name='name',
                context=(ILocationManagerTarget, IAdminLayer, LocationManagerSettingsTable),
                provides=IColumn)
class LocationManagerSettingsNameColumn(I18nColumn, GetAttrColumn):
    """Location manager settings name column"""

    _header = _("Setting type")
    attrName = 'name'
    weight = 10

    def getValue(self, item):
        value = super(LocationManagerSettingsNameColumn, self).getValue(item)
        translate = self.request.localizer.translate
        return translate(LOCATION_SETTINGS[value])


@adapter_config(name='show-hide', context=(ILocationManagerTarget, IAdminLayer, LocationManagerSettingsTable),
                provides=IColumn)
class LocationManagerSettingsVisibleColumn(VisibilitySwitcherColumn):
    """Location manager settings visibility column"""

    icon_hint = _("Set access or not to this setting in contents")

    weight = 80


@view_config(name='switch-location-setting-visibility.json', context=ILocationManagerTarget, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def switch_location_setting_visibility(request):
    """Set location setting visibility"""
    settings = ILocationManagerSettings(request.context)
    object_name = request.params.get('object_name')
    setting = settings.get(object_name)
    if setting is None:
        raise NotFound()
    setting.visible = not setting.visible
    settings[object_name] = setting  # for persistence...
    return {
        'status': 'success',
        'visible': setting.visible
    }


@adapter_config(name='fold-unfold', context=(ILocationManagerTarget, IAdminLayer, LocationManagerSettingsTable),
                provides=IColumn)
class LocationManagerSettingsFoldingColumn(AttributeSwitcherColumn):
    """Location manager settings folding column"""

    switch_attribute = 'unfolded'

    on_icon_class = 'fa fa-fw fa-folder-open-o'
    off_icon_class = 'fa fa-fw fa-folder-o'

    icon_hint = _("Pre-unfold or not this setting in contents")

    weight = 90


@view_config(name='switch-location-setting-folding.json', context=ILocationManagerTarget, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def switch_location_setting_folding(request):
    """Set location setting folding"""
    settings = ILocationManagerSettings(request.context)
    object_name = request.params.get('object_name')
    setting = settings.get(object_name)
    if setting is None:
        raise NotFound()
    setting.unfolded = not setting.unfolded
    settings[object_name] = setting  # for persistence...
    return {
        'status': 'success',
        'unfolded': setting.unfolded
    }


@adapter_config(context=(ILocationManagerTarget, IAdminLayer, LocationManagerSettingsTable), provides=IValues)
class LocationManagerSettingsValuesAdapter(ContextRequestViewAdapter):
    """Location manager settings table values adapter"""

    @property
    def values(self):
        settings = ILocationManagerSettings(self.context)
        for key in LOCATION_SETTINGS_KEYS:
            value = settings.get(key)
            if value is None:
                value = settings[key] = LocationSetting(key)
            yield value


@viewlet_config(name='location-settings-table', view=LocationManagerSettingsView, layer=IAdminLayer,
                manager=IWidgetsSuffixViewletsManager)
class LocationManagerSettingsTableViewlet(TableViewlet):
    """Location manager settings table viewlet"""

    table_class = LocationManagerSettingsTable


@adapter_config(context=(ILocationManagerTarget, IAdminLayer, LocationManagerSettingsView), provides=IFormHelp)
class LocationManagerSettingsViewHelp(FormHelp):
    """Location settings view form help"""

    message = _("Location settings are not required for each type of content.\n\n"
                "This form allows you to define which settings are available for this content type, and if so "
                "which ones are unfolded by default.")
    message_format = 'rest'
