#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

import json

from onf_website.component.location import ILocationInfo
from pyams_content.shared.logo import IWfLogo
from pyams_default_theme.interfaces import IContentNavigationIllustration, IContentTitle
from pyams_file.interfaces import IThumbnails
from pyams_i18n.interfaces import II18n
from pyams_utils.interfaces import ICacheKeyValue
from pyams_utils.url import absolute_url, canonical_url, relative_url


class BaseMapMarkersRendererMixin:
    """Base map markers renderer mixin"""

    def get_url(self, target):
        if IWfLogo.providedBy(target):
            if target.target is None:
                return target.url
            target = target.target
        if getattr(self.settings, 'force_canonical_url', False):
            return canonical_url(target, self.request)
        return relative_url(target, self.request)

    def get_markers(self, results, marker_image=None):
        if not marker_image:
            icon = {
                'url': '/--static--/onf_website/images/leaflet-marker.png',
                'size': [40, 53],
                'anchor': [20, 52]
            }
        else:
            marker_size = marker_image.get_image_size()
            icon = {
                'url': absolute_url(marker_image, self.request),
                'size': marker_size,
                'anchor': [marker_size[0] / 2, marker_size[1] - 1]
            }
        config = {
            'icon': icon,
            'clusterClass': 'onfCluster',
            'tooltipClass': 'onfTooltip',
            'markers': []
        }
        append = config['markers'].append
        registry = self.request.registry
        display_illustrations = self.renderer_settings.display_illustrations
        for result in results:
            location_info = ILocationInfo(result, None)
            if (location_info is None) or not location_info.locations:
                continue
            for location in location_info.locations:
                position = location.to_json()
                if position is None:
                    continue
                href = self.get_url(result)
                if href is None:
                    continue
                marker = {
                    'id': ICacheKeyValue(result),
                    'label': registry.queryMultiAdapter((result, self.request),
                                                        IContentTitle),
                    'href': href,
                    'point': position
                }
                if display_illustrations:
                    illustration = registry.queryMultiAdapter((result, self.request),
                                                              IContentNavigationIllustration)
                    if illustration:
                        img = II18n(illustration).query_attribute('data', request=self.request)
                        if img:
                            marker['img'] = {
                                'src': absolute_url(img, request=self.request,
                                                    view_name='++thumb++pano:w256'),
                                'w': 256,
                                'h': 144
                            }
                append(marker)
        return json.dumps(config)

    def get_map_configuration(self):
        return json.dumps(self.renderer_settings.configuration)
