#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid_rpc.jsonrpc import jsonrpc_method
from sqlalchemy.orm import object_session
from sqlalchemy.sql import func

from onf_website.reference.forest.model.foret import Foret, ForetGeom, \
    PARENT_SESSION as FORET_SESSION
from onf_website.reference.insee.model import Commune, CommuneGeom
from pyams_alchemy.engine import get_user_session
from pyams_gis.interfaces import WGS84
from pyams_gis.transform import transform
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION


__docformat__ = 'restructuredtext'


@jsonrpc_method(endpoint='onf_website',
                permission=VIEW_SYSTEM_PERMISSION)
def findLocation(request, query):

    def get_locations():
        for foret in Foret.find(query).order_by(Foret.libelle_strict, Foret.libelle_usage):
            yield {
                'type': 'foret',
                'id': foret.token,
                'text': foret.title
            }
        for commune in Commune.find(query).order_by(Commune.nccenr):
            yield {
                'type': 'commune',
                'id': commune.token,
                'text': commune.title
            }

    return list(get_locations())


@jsonrpc_method(endpoint='onf_website',
                permission=VIEW_SYSTEM_PERMISSION)
def getLocationCoords(request, source, to_srid):
    """Get GPS coordinates matching given object"""
    type = source['type']
    target = None
    if type == 'foret':
        target = Foret.find_by_id(source['id']) \
                      .join(ForetGeom, Foret.id_nat_frt==ForetGeom.id_nat_frt) \
                      .first()
    elif type == 'commune':
        target = Commune.find_by_insee_code(source['id']) \
                        .join(CommuneGeom, Commune.code==CommuneGeom.code) \
                        .first()
    if target is not None:
        session = object_session(target)
        if type == 'foret':
            centroid = session.scalar(func.ST_PointOnSurface(target.ref_geom.geom))
        else:
            centroid = session.scalar(func.ST_Centroid(target.ref_geom.geom))
        geojson = session.scalar(func.ST_AsGeoJSON(target.ref_geom.geom))
        position = transform({
            'longitude': session.scalar(func.ST_X(centroid)),
            'latitude': session.scalar(func.ST_Y(centroid))
        }, WGS84, int(to_srid))
        return {
            'type': source['type'],
            'id': source['id'],
            'position': {
                'lon': position['point']['longitude'],
                'lat': position['point']['latitude']
            },
            'srid': position['srid'],
            'geometry': json.loads(geojson)
        }


@jsonrpc_method(endpoint='onf_website',
                permission=VIEW_SYSTEM_PERMISSION)
def getForestAt(request, position, srid):
    """Get forest at given position"""
    session = get_user_session(FORET_SESSION)
    target = transform(position, int(srid), WGS84)
    marker = 'POINT({0} {1})'.format(target['point']['longitude'], target['point']['latitude'])
    result = session.query(Foret, func.ST_AsGeoJSON(ForetGeom.geom)) \
                    .join(ForetGeom, Foret.id_nat_frt==ForetGeom.id_nat_frt) \
                    .filter(ForetGeom.geom.ST_Intersects(func.ST_GeomFromText(marker, WGS84))) \
                    .first()
    if result is not None:
        foret, geojson = result
        return {
            'type': 'foret',
            'id': foret.token,
            'text': foret.title,
            'srid': target['srid'],
            'geometry': json.loads(geojson)
        }
